<?php

namespace AppBundle\Security;

use Symfony\Component\HttpFoundation\RedirectResponse;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\Session\SessionInterface;
use Symfony\Component\Routing\RouterInterface;
use Symfony\Component\Security\Core\Authentication\Token\TokenInterface;
use Symfony\Component\Security\Core\Authorization\AuthorizationCheckerInterface;
use Symfony\Component\Security\Http\Authentication\AuthenticationSuccessHandlerInterface;

/**
 * Override the authentication handler.
 * 
 * @see http://www.reecefowell.com/2011/10/26/redirecting-on-loginlogout-in-symfony2-using-loginhandlers/
 */
class LoginSuccessHandler implements AuthenticationSuccessHandlerInterface
{
    protected $authorizationChecker;
    protected $router;
    protected $session;

    /**
     * Constructor
     * 
     * @param AuthorizationCheckerInterface $authorizationChecker 
     * @param RouterInterface $router 
     */
    public function __construct(
        AuthorizationCheckerInterface $authorizationChecker, 
        RouterInterface $router, 
        SessionInterface $session
    )
    {
        $this->authorizationChecker = $authorizationChecker;
        $this->router = $router;
        $this->session = $session;
    }

    /**
     *
     * @param Request $request
     * @param TokenInterface $token
     * @return RedirectResponse
     */
    public function onAuthenticationSuccess(Request $request, TokenInterface $token)
    {
        // Get $redirectUrl if params _return exists.
        $redirectUrl = null;
        
        if ($this->session->has('_return')){
            $_return = $this->session->get('_return'); 
        }
        else {
            $_return = $request->request->get('_return');
        }

        if (isset($_return)) {
            // Decode params
            $_return = base64_decode($_return, true);

            if (filter_var($_return, FILTER_VALIDATE_URL)) {
                // Security - Check if redirect is local url 
                if (strpos($_return, $request->getSchemeAndHttpHost()) === 0) {
                    $redirectUrl = $_return;
                }
            }
        }
        
        $this->session->remove('_return');
        
        if ($this->authorizationChecker->isGranted('ROLE_ADMIN_FULL')) {
            $response = new RedirectResponse($this->router->generate('admin_homepage'));
        } elseif ($this->authorizationChecker->isGranted('ROLE_ADMIN_ITDLC') || $this->authorizationChecker->isGranted('ROLE_ADMIN_PEE')) {
            $response = new RedirectResponse($this->router->generate('admin_list_user'));
        } elseif (!empty($redirectUrl)) {
            $response = new RedirectResponse($redirectUrl);
        } elseif ($this->authorizationChecker->isGranted('ROLE_USER')) {
            $response = new RedirectResponse($this->router->generate('espace_perso'));
        } else {
            // redirect the user to the page he was before the login process began.
            $refererUrl = $request->headers->get('referer');

            $response = new RedirectResponse($refererUrl);
        }

        return $response;
    }
}
