<?php

namespace AppBundle\Entity;

use Doctrine\ORM\Mapping as ORM;

use Cocur\Slugify\Slugify;
use Symfony\Component\HttpFoundation\File\UploadedFile;
use Symfony\Component\Validator\Constraints as Assert;

/**
 * Testimonial
 *
 * @ORM\Table(name="testimonial")
 * @ORM\Entity(repositoryClass="AppBundle\Repository\TestimonialRepository")
 */
class Testimonial
{
    /**
     * @var int
     *
     * @ORM\Column(name="id", type="integer")
     * @ORM\Id
     * @ORM\GeneratedValue(strategy="AUTO")
     */
    private $id;

    /**
     * @var string
     *
     * @ORM\Column(name="witness", type="string", length=255)
     */
    private $witness;

    /**
     * @var string
     *
     * @ORM\Column(name="occupation", type="string", length=255, nullable=true)
     */
    private $occupation;

    /**
     * @var \String
     *
     * @ORM\Column(name="picture", type="string", length=255, nullable=true)
     */
    private $picture;

    /**
     * @var string
     *
     * @ORM\Column(name="content", type="text", nullable=true)
     */
    private $content;

    /**
     * @var bool
     *
     * @ORM\Column(name="is_active", type="boolean", nullable=false)
     */
    private $isActive = true;

    /**
     * @var \DateTime
     *
     * @ORM\Column(name="created", type="datetime")
     */
    private $created;

    /**
     * @var \DateTime|null
     *
     * @ORM\Column(name="updated", type="datetime", nullable=true)
     */
    private $updated;


    public function __tostring()
    {
        return $this->title;
    }

    /**
     * Get id.
     *
     * @return int
     */
    public function getId()
    {
        return $this->id;
    }

    /**
     * Set Witness name.
     *
     * @param string $name
     *
     * @return Testimonial
     */
    public function setWitness($name)
    {
        $this->witness = $name;

        return $this;
    }

    /**
     * Get witness name.
     *
     * @return string
     */
    public function getWitness()
    {
        return $this->witness;
    }

    /**
     * Set Witness Occupation.
     *
     * @param string $occupation
     *
     * @return Testimonial
     */
    public function setOccupation($occupation)
    {
        $this->occupation = $occupation;

        return $this;
    }

    /**
     * Get Witness Occupation.
     *
     * @return string
     */
    public function getOccupation()
    {
        return $this->occupation;
    }

    /**
     * Set picture.
     *
     * @param string $picture
     *
     * @return Testimonial
     */
    public function setPicture($picture)
    {        
        if (null === $picture) {
            $this->tempPicture = $this->getAbsolutePicture();
        }
        
        $this->picture = $picture;

        return $this;
    }

    /**
     * Get picture.
     *
     * @return string
     */
    public function getPicture()
    {
        return $this->picture;
    }

    /**
     * Set content.
     *
     * @param string $content
     *
     * @return Testimonial
     */
    public function setContent($content)
    {
        $this->content = $content;

        return $this;
    }

    /**
     * Get content.
     *
     * @return string
     */
    public function getContent()
    {
        return $this->content;
    }

    /**
     * Set isActive.
     *
     * @param bool $isActive
     *
     * @return Testimonial
     */
    public function setIsActive($isActive)
    {
        $this->isActive = $isActive;

        return $this;
    }

    /**
     * Get isActive.
     *
     * @return bool
     */
    public function getIsActive()
    {
        return $this->isActive;
    }

    /**
     * Set created.
     *
     * @param \DateTime $created
     *
     * @return Testimonial
     */
    public function setCreated($created)
    {
        $this->created = $created;

        return $this;
    }

    /**
     * Get created.
     *
     * @return \DateTime
     */
    public function getCreated()
    {
        return $this->created;
    }

    /**
     * Set updated.
     *
     * @param \DateTime|null $updated
     *
     * @return Testimonial
     */
    public function setUpdated($updated = null)
    {
        $this->updated = $updated;

        return $this;
    }

    /**
     * Get updated.
     *
     * @return \DateTime|null
     */
    public function getUpdated()
    {
        return $this->updated;
    }



    /****
     * 
     * FILE UPLOAD
     * 
     *****/

    private $tempPicture;

     /**
     * @Assert\Image(
     *      maxWidth = 400,
     *      maxWidthMessage = "Lʼimage ne doit pas dépasser 400 pixels de large",
     *      maxHeight = 400,
     *      maxHeightMessage = "Lʼimage ne doit pas dépasser 400 pixels de haut",
     *      maxSizeMessage = "Lʼimage ne doit pas dépasser 2Mb.",
     *      maxSize = "2000000",
     *      mimeTypes = {"image/jpg", "image/jpeg", "image/gif", "image/png"},
     *      mimeTypesMessage = "Les images doivent être au format JPG, GIF ou PNG."
     * )
     */
    private $filePicture;

    /**
     * Sets file.
     *
     * @param UploadedFile $filePicture
     */
    public function setFilePicture(UploadedFile $filePicture = null)
    {
        $this->filePicture = $filePicture;

        // check if we have an old image path
        if (is_file($this->getAbsolutePicture())) {
            // store the old name to delete after the update
            $this->tempPicture = $this->getAbsolutePicture();
        }
    }

     /**
     * Get UploadedFile Picture
     *
     * @return UploadedFile
     */
    public function getFilePicture()
    {
        return $this->filePicture;   
    }

    public function getAbsolutePicture()
    {
        return  null === $this->picture
                ? null
                : $this->getUploadRootDir() . DIRECTORY_SEPARATOR . $this->picture;
    }

    public function getWebPicture()
    {
        return null === $this->picture
                ? null
                : $this->getUploadDir() . DIRECTORY_SEPARATOR . $this->picture;
    }

    protected function getUploadRootDir()
    {
        // the absolute directory path where uploaded
        // documents should be saved
        return __DIR__
            . DIRECTORY_SEPARATOR
            . join(DIRECTORY_SEPARATOR, ['..', '..', '..', 'web'])
            . DIRECTORY_SEPARATOR
            . $this->getUploadDir()
        ;
    }

    protected function getUploadDir()
    {
        // get rid of the __DIR__ so it doesn't screw up
        // when displaying uploaded doc/image in the view.
        return join(DIRECTORY_SEPARATOR, ['uploads', 'testimonial']);
    }

    /**
     * @ORM\PrePersist()
     * @ORM\PreUpdate()
     */
    public function preUploadFilePicture()
    {   
        $uniqueIdentifier = sha1(uniqid(mt_rand(), true));
        $slugify = new Slugify();

        if (null !== $this->getFilePicture()) {
            $filenameWithoutExtension = substr($this->getFilePicture()->getClientOriginalName(), 0, -1 * (strlen($this->getFilePicture()->getClientOriginalExtension()) + 1));

            $this->picture = $uniqueIdentifier . '-' . $slugify->slugify($filenameWithoutExtension) . '.' . $this->getFilePicture()->getClientOriginalExtension();
        }
    }

    /**
     * @ORM\PostPersist()
     * @ORM\PostUpdate()
     */
    public function uploadFilePicture()
    {
        if (null === $this->getFilePicture()) {
            return;
        }

        //Delete old file
        if (isset($this->tempPicture)) {
            if (is_file($this->tempPicture)) {
                unlink($this->tempPicture);
            }
        }

        //Move new file
        if (isset($this->picture) && is_string($this->picture)) {
            $this->getFilePicture()->move(
                $this->getUploadRootDir(),
                $this->picture
            );
        }

        $this->setFilePicture(null);
    }

    /**
     * @ORM\PreRemove()
     */
    public function storeFilenameForRemove()
    {
        $this->tempPicture = $this->getAbsolutePicture();
    }

    /**
     * @ORM\PostRemove()
     */
    public function removeUpload()
    {
        if (isset($this->tempPicture) && is_file($this->tempPicture)) {
            unlink($this->tempPicture);
        }
    }
}
