<?php

namespace AppBundle\Controller\Admin;

use AppBundle\Entity\DemandePEE;
use AppBundle\Entity\User;
use AppBundle\Enum\DemandePEEState;
use AppBundle\Enum\OperationPEEState;
use AppBundle\Form\Type\PEE\DemandePEEType;
use AppBundle\Translation\ConstTranslator;
use Doctrine\ORM\EntityManagerInterface;
use Doctrine\ORM\Query;
use Doctrine\ORM\Tools\Pagination\Paginator;
use Sensio\Bundle\FrameworkExtraBundle\Configuration\Route;
use Symfony\Bundle\FrameworkBundle\Controller\Controller;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\Response;
use Symfony\Component\HttpFoundation\Session\SessionInterface;

class DemandePEEController extends Controller
{
    /**
     * @var EntityManagerInterface
     */
    private $em;

    /**
     * @var SessionInterface
     */
    private $session;

    /**
     * @var ConstTranslator
     */
    private $utils;
    
    public function __construct(EntityManagerInterface $em, SessionInterface $session, ConstTranslator $utils)
    {
        $this->em = $em;
        $this->session = $session;
        $this->utils = $utils;
    }

    /**
     * Liste des demandes PEE
     *
     * @Route(
     *      "/demande-pee/list/{page}",
     *      requirements = {"page": "\d+"},
     *      name = "admin_demande_pee_liste"
     * )
     *
     * @param Request $request
     * @param int $page
     * @return Response
     */
    public function indexAction(Request $request, int $page = 1)
    {
        $resultsMaxPerPage = $this->getParameter('results.admin.max_per_page');

        // Mise à jour de la session avec les nouvelles valeurs
        $this->session->set($request->attributes->get('_route'), array(
            'date' => $request->query->get('date'),
            'page' => $page,
            'lastname' => $request->query->get('lastname'),
            'email' => $request->query->get('email'),
            'academy' => $request->query->get('academy'),
            'dState' => $request->query->get('dState'),
            'oState' => $request->query->get('oState'),
        ));

        $academies = $this->em->getRepository(User::class)->findAcademies(Query::HYDRATE_ARRAY);
        $dStates = $this->utils->trans(DemandePEEState::class);
        $oStates = $this->utils->trans(OperationPEEState::class);

        $qb = $this->em->getRepository(DemandePEE::class)->findFromRequest($request, false);
        $qb
            ->setFirstResult(($page - 1) * $resultsMaxPerPage)
            ->setMaxResults($resultsMaxPerPage);
        $demandesPee = new Paginator($qb);

        return $this->render('Admin/Ope/PEE/indexDemande.html.twig', array(
            'academies' => $academies,
            'demandes' => $demandesPee,
            'nbPages' => ceil(count($demandesPee) / $resultsMaxPerPage),
            'dStates' => $dStates,
            'oStates' => $oStates,
        ));
    }

    /**
     * Modification d'une demande PEE
     *
     * @Route(
     *      "/demande-pee/{id}/edit",
     *      name = "admin_demande_pee_edit"
     * )
     *
     * @param Request $request
     * @param DemandePEE $demandePee
     * @return \Symfony\Component\HttpFoundation\RedirectResponse|Response
     * @throws \ReflectionException
     */
    public function editAction(Request $request, DemandePEE $demandePee)
    {
        $countDemandesByState = $this->em->getRepository(DemandePEE::class)->countByState($demandePee->getOperationPEE()->getId());

        $form = $this->createForm(DemandePEEType::class, $demandePee);
        $form->handleRequest($request);

        if ($form->isSubmitted() && $form->isValid()) {

            $this->em->persist($demandePee);
            $this->em->flush();

            $this->addFlash('success', 'Demande PEE mise à jour');

            return $this->redirectToRoute('admin_demande_pee_edit', [
                'id' => $demandePee->getId(),
            ]);
        }

        return $this->render('Admin/Ope/PEE/editDemande.html.twig', [
            'form' => $form->createView(),
            'ope' => $demandePee,
            'countDemandesByState' => $countDemandesByState,
        ]);
    }

    /**
     * Supprime une demande PEE
     *
     * @Route(
     *      "/demande-pee/{id}/delete",
     *      name = "admin_demande_pee_delete"
     * )
     *
     * @param DemandePEE $demandePee
     * @return Response
     */
    public function deleteAction(DemandePEE $demandePee)
    {
        $this->em->remove($demandePee);
        $this->em->flush();

        $this->addFlash('success', 'Elément supprimé');

        return new Response();
    }
}
