<?php

namespace AppBundle\Controller\Admin;

use AppBundle\Entity\Content;
use AppBundle\Entity\Menu;
use AppBundle\Entity\Module;
use AppBundle\Entity\TypeTemplate;
use AppBundle\Enum\TypeTemplateTitle;
use AppBundle\Form\Type\ActuContentType;
use AppBundle\Form\Type\BlocContentType;
use AppBundle\Form\Type\ContentType;
use AppBundle\Form\Type\SimpleContentType;
use AppBundle\Form\Type\Menu\SimpleMenuType;
use AppBundle\Form\Type\Menu\ListeMenuType;
use AppBundle\Form\Type\Menu\ListChildMenuType;
use AppBundle\Form\Type\Menu\PartnerMenuType;
use Cocur\Slugify\Slugify;
use Doctrine\ORM\EntityManagerInterface;
use Doctrine\ORM\Tools\Pagination\Paginator;
use Sensio\Bundle\FrameworkExtraBundle\Configuration\Route;
use Symfony\Bundle\FrameworkBundle\Controller\Controller;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\Session\SessionInterface;

class ContentController extends Controller
{
    /**
     * @var EntityManagerInterface
     */
    private $em;

    /**
     * @var SessionInterface
     */
    private $session;
    
    public function __construct(EntityManagerInterface $em, SessionInterface $session)
    {
        $this->em = $em;
        $this->session = $session;
    }

    /**
     * @Route(
     *      "/menu/{menu_type}/add-page-simple",
     *      name = "admin_content_add_page_simple"
     * )
     *
     * @param Request $request
     * @param string menu_type
     * @return \Symfony\Component\HttpFoundation\RedirectResponse|\Symfony\Component\HttpFoundation\Response
     */
    public function addPageSimpleAction(Request $request, string $menu_type = 'main')
    {
        $slugify = new Slugify();

        // Formulaire page simple 
        $content = new Content;
        $content->setIsModule(false);
        $content->setIsMain(true);
        $content->setIsActu(false);
        $formContent = $this->createForm(ContentType::class, $content);
        $formContent->handleRequest($request);

        // Formulaire menu article 
        $menu = new Menu;
        $menu->setMenuType($menu_type);
        $menu->setTypeTemplate($this->em->getRepository(TypeTemplate::class)->findOneByTitle(TypeTemplateTitle::SIMPLE));
        $formMenu = $this->createForm(SimpleMenuType::class, $menu);
        $formMenu->handleRequest($request);

        if ($formContent->isSubmitted() && $formContent->isValid() && $formMenu->isSubmitted() && $formMenu->isValid()) {
            // On enregistre le menu de la page simple 
            $menu->setSlug($slugify->slugify($menu->getTitle()));
            $this->em->persist($menu);

            // On enregistre le contenu de la page simple 
            $content->setMenu($menu);
            $this->em->persist($content);

            $this->em->flush();

            $this->addFlash('success', 'Page texte ajouté');

            return $this->redirectToRoute('admin_content_edit_page_simple', array(
                'menu' => $menu->getId(),
                'content' => $content->getId(),
                'mode' => 'edit',
                'menu_type' => $menu->getMenuType(),
            ));
        }

        return $this->render('Admin/Content/editPage.html.twig', array(
            'form' => $formContent->createView(),
            'form_menu' => $formMenu->createView(),
            'menu' => $menu,
        ));
    }

    /**
     *
     * @Route(
     *      "/menu/{menu_type}/edit-page-simple/{menu}/{content}",
     *      name = "admin_content_edit_page_simple"
     * )
     *
     * @param Request $request
     * @param Menu menu
     * @param Content $content
     * @param string menu_type
     * @return \Symfony\Component\HttpFoundation\RedirectResponse|\Symfony\Component\HttpFoundation\Response
     */
    public function editPageSimpleAction(Request $request, Menu $menu, Content $content, string $menu_type = 'main')
    {
        $slugify = new Slugify();

        /* $querymodules = $this->em->createQuery(
            "SELECT c FROM AppBundle:Content c WHERE c.menu=:c_menu and c.is_module=:c_is_module ORDER BY c.position ASC, c.id ASC"
        )->setParameters(array(
            'c_menu' => $menu->getId(),
            'c_is_module' => true,
        ));
        $modules = $querymodules->getResult();
        */ 

        $modules = $this->em->getRepository(Module::class)->findByMenu(
            $menu->getId(), 
            array(
                //'isActive' => true,
                'notLikePosition' => array("home_%", "login_%", "modal_%", "menu_%", "social", "newsletter")
            )
        );

        $formContent = $this->createForm(ContentType::class, $content);
        $formContent->handleRequest($request);
        
        $formMenu = $this->createForm(SimpleMenuType::class, $menu);
        $formMenu->handleRequest($request);

        if ($formContent->isSubmitted() && $formContent->isValid() && $formMenu->isSubmitted() && $formMenu->isValid()) {
            // Update menu
            $menu->setSlug($slugify->slugify($menu->getTitle()));
            $this->em->persist($menu);

            // Update content
            if ($request->get('remove_picture')) {
                $content->setPicture(null);
            }
            $this->em->persist($content);

            $this->em->flush();

            $this->addFlash('success', 'Page simple modifiée');

            return $this->redirectToRoute('admin_content_edit_page_simple', array(
                'menu' => $menu->getId(),
                'content' => $content->getId(),
                'menu_type' => $menu->getMenuType(),
            ));
        }

        return $this->render('Admin/Content/editPage.html.twig', array(
            'form' => $formContent->createView(),
            'form_menu' => $formMenu->createView(),
            'modules' => $modules,
            'menu' => $menu,
            'content' => $content,
        ));
    }

    /**
     *
     * @Route(
     *      "/menu/{menu_type}/add-page-liste",
     *      name = "admin_content_add_page_liste"
     * )
     *
     * @param Request $request
     * @param string menu_type
     * @return \Symfony\Component\HttpFoundation\RedirectResponse|\Symfony\Component\HttpFoundation\Response
     */
    public function addPageListeAction(Request $request, string $menu_type = 'main')
    {
        $slugify = new Slugify();

        // Formulaire menu article 
        $menu = new Menu;
        $menu->setMenuType($menu_type);
        $menu->setTypeTemplate($this->em->getRepository(TypeTemplate::class)->findOneByTitle(TypeTemplateTitle::LIST));
        $formMenu = $this->createForm(SimpleMenuType::class, $menu);
        $formMenu->handleRequest($request);

        // Formulaire page simple 
        $content = new Content;
        $content->setIsModule(false);
        $content->setIsMain(true);
        $content->setIsActu(false);
        $content->setMenu($menu);
        $formContent = $this->createForm(ContentType::class, $content);
        $formContent->handleRequest($request);

        $formMenuParams = $this->createForm(ListeMenuType::class, $menu->getParams());
        $formMenuParams->handleRequest($request);

        if ($formContent->isSubmitted() && $formContent->isValid() && $formMenu->isSubmitted() && $formMenu->isValid() && $formMenuParams->isSubmitted() && $formMenuParams->isValid()) {
            // On rengistre les paramètres du Menu
            $params = (object) $formMenuParams->getData();
            $menu->setParams($params);

            // On enregistre le menu de la page simple 
            $menu->setSlug($slugify->slugify($menu->getTitle()));
            $this->em->persist($menu);

            // On enregistre le contenu de la page simple 
            $this->em->persist($content);

            $this->em->flush();

            $this->addFlash('success', 'Page liste ajouté');

            return $this->redirectToRoute('admin_content_edit_page_liste', array(
                'menu' => $menu->getId(),
                'content' => $content->getId(),
                'menu_type' => $menu->getMenuType(),
            ));
        }

        return $this->render('Admin/Content/editPage.html.twig', array(
            'form' => $formContent->createView(),
            'form_menu' => $formMenu->createView(),
            'form_menu_params' => $formMenuParams->createView(),
            'menu' => $menu,
        ));
    }

    /**
     *
     * @Route(
     *      "/menu/{menu_type}/edit-page-liste/{menu}/{content}",
     *      name = "admin_content_edit_page_liste"
     * )
     *
     * @param Request $request
     * @param Menu $menu
     * @param Content $content
     * @param string menu_type
     * @return \Symfony\Component\HttpFoundation\RedirectResponse|\Symfony\Component\HttpFoundation\Response
     */
    public function editPageListeAction(Request $request, Menu $menu, Content $content, string $menu_type = 'main')
    {
        $slugify = new Slugify();

        $querycontents = $this->em
            ->createQuery(
                "SELECT c FROM AppBundle:Content c
                    WHERE c.menu = :c_menuid
                        AND c.is_module = :c_ismodule
                        AND c.is_main = :c_ismain
                    ORDER BY c.position ASC, c.id ASC"
            )
            ->setParameters(array(
                'c_menuid' => $menu->getId(),
                'c_ismodule' => false,
                'c_ismain' => false,
            ));
        $contenus = $querycontents->getResult();

        /* $querymodules = $this->em
            ->createQuery(
                "SELECT c FROM AppBundle:Content c
                    WHERE c.menu = :menuid
                    AND c.is_module = :module
                    ORDER BY c.position ASC, c.id ASC"
            )
            ->setParameters(array(
                'menuid' => $menu->getId(),
                'module' => true,
            ));
        $modules = $querymodules->getResult(); */

        $modules = $this->em->getRepository(Module::class)->findByMenu(
            $menu->getId(), 
            array(
                //'isActive' => true,
                'notLikePosition' => array("home_%", "login_%", "modal_%", "menu_%", "social", "newsletter")
            )
        );

        $formContent = $this->createForm(ContentType::class, $content);
        $formContent->handleRequest($request);

        $formMenu = $this->createForm(SimpleMenuType::class, $menu);
        $formMenu->handleRequest($request);

        $defaultParams = array('layout' => null); // not break legacy
        $menuParams = (array) $menu->getParams();

        $menuParams = array_merge($defaultParams, $menuParams);

        $formMenuParams = $this->createForm(ListeMenuType::class, $menuParams);
        $formMenuParams->handleRequest($request);

        if ($formContent->isSubmitted() && $formContent->isValid() && $formMenu->isSubmitted() && $formMenu->isValid() && $formMenuParams->isSubmitted() && $formMenuParams->isValid()) {
            //Force params to null for update | Bug Doctrine
            $menu->setParams(null);
            $this->em->persist($menu);
            $this->em->flush();

            // Update menu
            $menu->setSlug($slugify->slugify($menu->getTitle()));
            $params = (object) $formMenuParams->getData();
            $menu->setParams($params);
            $this->em->persist($menu);

            $this->em->persist($content);

            $this->em->flush();

            $this->addFlash('success', 'Page liste modifiée');

            return $this->redirectToRoute('admin_content_edit_page_liste', array(
                'menu' => $menu->getId(),
                'content' => $content->getId(),
                'menu_type' => $menu->getMenuType(),
            ));
        }

        return $this->render('Admin/Content/editPage.html.twig', array(
            'form' => $formContent->createView(),
            'form_menu' => $formMenu->createView(),
            'form_menu_params' => $formMenuParams->createView(),
            'modules' => $modules,
            'menu' => $menu,
            'content' => $content,
            'contenus' => $contenus,
        ));
    }

    /**
     *
     * @Route(
     *      "/menu/{menu_type}/add-page-presentation",
     *      name = "admin_content_add_page_presentation"
     * )
     *
     * @param Request $request
     * @param string menu_type
     * @return \Symfony\Component\HttpFoundation\RedirectResponse|\Symfony\Component\HttpFoundation\Response
     */
    public function addPagePresentationAction(Request $request, string $menu_type = 'main')
    {
        $slugify = new Slugify();

        $content = new Content;
        $content->setIsModule(false);
        $content->setIsMain(true);
        $content->setIsActu(false);

        $menu = new Menu;
        $menu->setMenuType($menu_type);
        $menu->setTypeTemplate($this->em->getRepository(TypeTemplate::class)->findOneByTitle(TypeTemplateTitle::PRESENTATION));

        // Formulaire page simple 
        $formContent = $this->createForm(ContentType::class, $content);
        $formContent->handleRequest($request);

        // Formulaire menu article 
        $formMenu = $this->createForm(SimpleMenuType::class, $menu);
        $formMenu->handleRequest($request);

        if ($formContent->isSubmitted() && $formContent->isValid() && $formMenu->isSubmitted() && $formMenu->isValid()){
            // On enregistre le menu de la page simple 
            $menu->setSlug($slugify->slugify($menu->getTitle()));
            $this->em->persist($menu);

            // On enregistre le contenu de la page simple 
            $content->setMenu($menu);
            $this->em->persist($content);

            $this->em->flush();

            $this->addFlash('success', 'Page presentation ajouté');

            return $this->redirectToRoute('admin_content_edit_page_presentation', array(
                'menu' => $menu->getId(),
                'content' => $content->getId(),
                'menu_type' => $menu->getMenuType(),
            ));
        }

        return $this->render('Admin/Content/editPage.html.twig', array(
            'form' => $formContent->createView(),
            'form_menu' => $formMenu->createView(),
            'menu' => $menu,
        ));
    }

    /**
     *
     * @Route(
     *      "/menu/{menu_type}/edit-page-presentation/{menu}/{content}",
     *      name = "admin_content_edit_page_presentation"
     * )
     *
     * @param Request $request
     * @param Menu $menu
     * @param Content $content
     * @param string menu_type
     * @return \Symfony\Component\HttpFoundation\RedirectResponse|\Symfony\Component\HttpFoundation\Response
     */
    public function editPagePresentationAction(Request $request, Menu $menu, Content $content, string $menu_type = 'main')
    {
        $slugify = new Slugify();

        $medias = $this->em->getRepository('AppBundle:Media')->findByMenu($menu->getId(), array('is_active' => false));

        $req_contenus = $this->em
            ->createQuery(
                "SELECT c
                FROM AppBundle:Content c
                WHERE c.menu = :menuid
                    AND c.is_module = :module
                    AND c.is_main = :main
                ORDER BY c.position ASC, c.id ASC"
            )
            ->setParameters(array(
                'menuid' => $menu->getId(),
                'module' => false,
                'main' => false,
            ));
        $contenus = $req_contenus->getResult();

        /* $querymodules = $this->em
            ->createQuery(
                "SELECT c
                FROM AppBundle:Content c
                WHERE c.menu = :menuid
                    AND c.is_module = :module
                ORDER BY c.position ASC, c.id ASC"
            )
            ->setParameters(array(
                'menuid' => $menu->getId(),
                'module' => true,
            ));
        $modules = $querymodules->getResult(); */

        $modules = $this->em->getRepository(Module::class)->findByMenu(
            $menu->getId(), 
            array(
                //'isActive' => true,
                'notLikePosition' => array("home_%", "login_%", "modal_%", "menu_%", "social", "newsletter")
            )
        );

        $formContent = $this->createForm(ContentType::class, $content);
        $formContent->handleRequest($request);

        $formMenu = $this->createForm(SimpleMenuType::class, $menu);
        $formMenu->handleRequest($request);

        if ($formContent->isSubmitted() && $formContent->isValid() && $formMenu->isSubmitted() && $formMenu->isValid()) {
            $menu->setSlug($slugify->slugify($menu->getTitle()));
            $this->em->persist($menu);

            $this->em->persist($content);

            $this->em->flush();

            $this->addFlash('success', 'Page présentation modifiée');

            return $this->redirectToRoute('admin_content_edit_page_presentation', array(
                'menu' => $menu->getId(),
                'content' => $content->getId(),
                'menu_type' => $menu->getMenuType(),
            ));
        }

        return $this->render('Admin/Content/editPage.html.twig', array(
            'form' => $formContent->createView(),
            'form_menu' => $formMenu->createView(),
            'modules' => $modules,
            'medias' => $medias,
            'menu' => $menu,
            'content' => $content,
            'contenus' => $contenus,
        ));
    }

    /**
     *
     * @Route(
     *      "/menu/{menu_type}/add-list-child",
     *      name = "admin_content_add_list_child"
     * )
     *
     * @param Request $request
     * @param string menu_type
     * @return \Symfony\Component\HttpFoundation\RedirectResponse|\Symfony\Component\HttpFoundation\Response
     */
    public function addListChildAction(Request $request, string $menu_type = 'main')
    {
        $slugify = new Slugify();

        $content = new Content;
        $content->setIsModule(false);
        $content->setIsMain(true);
        $content->setIsActu(false);

        $menu = new Menu;
        $menu->setMenuType($menu_type);
        $menu->setTypeTemplate($this->em->getRepository(TypeTemplate::class)->findOneByTitle(TypeTemplateTitle::LIST_CHILD));

        // Formulaire page simple 
        $formContent = $this->createForm(ContentType::class, $content);
        $formContent->handleRequest($request);

        // Formulaire menu article 
        $formMenu = $this->createForm(SimpleMenuType::class, $menu);
        $formMenu->handleRequest($request);

        $formMenuParams = $this->createForm(ListChildMenuType::class, $menu->getParams());
        $formMenuParams->handleRequest($request);

        if ($formContent->isSubmitted() && $formContent->isValid() && $formMenu->isSubmitted() && $formMenu->isValid() && $formMenuParams->isSubmitted() && $formMenuParams->isValid()) {
            // On rengistre les paramètres du Menu
            $params = (object) $formMenuParams->getData();
            $menu->setParams($params);

            // On enregistre le menu de la page simple 
            $menu->setSlug($slugify->slugify($menu->getTitle()));
            $this->em->persist($menu);

            // On enregistre le contenu de la page simple 
            $content->setMenu($menu);
            $this->em->persist($content);

            $this->em->flush();

            $this->addFlash('success', 'Page liste ajouté');

            return $this->redirectToRoute('admin_content_edit_list_child', array(
                'menu' => $menu->getId(),
                'content' => $content->getId(),
                'menu_type' => $menu->getMenuType(),
            ));
        }

        return $this->render('Admin/Content/editPage.html.twig', array(
            'form' => $formContent->createView(),
            'form_menu' => $formMenu->createView(),
            'form_menu_params' => $formMenuParams->createView(),
            'menu' => $menu,
        ));
    }

    /**
     *
     * @Route(
     *      "/menu/{menu_type}/edit-list-child/{menu}/{content}",
     *      name = "admin_content_edit_list_child"
     * )
     *
     * @param Request $request
     * @param Menu $menu
     * @param Content $content
     * @param string menu_type
     * @return \Symfony\Component\HttpFoundation\RedirectResponse|\Symfony\Component\HttpFoundation\Response
     */
    public function editListChildAction(Request $request, Menu $menu, Content $content, string $menu_type = 'main')
    {
        $slugify = new Slugify();

        /* $qb = $this->em->createQueryBuilder();
        $qb
            ->select('c')
            ->from('AppBundle:Content', 'c')
            ->where('c.menu = :menu_id')
            ->andWhere('c.is_module = :is_module')
            ->orderBy('c.position', 'asc')
            ->addOrderBy('c.id', 'asc')
            ->setParameters(array(
                'menu_id' => $menu->getId(),
                'is_module' => true,
            ));
        $modules = $qb->getQuery()->getResult(); */

        $modules = $this->em->getRepository(Module::class)->findByMenu(
            $menu->getId(), 
            array(
                //'isActive' => true,
                'notLikePosition' => array("home_%", "login_%", "modal_%", "menu_%", "social", "newsletter")
            )
        );

        $formContent = $this->createForm(ContentType::class, $content);
        $formContent->handleRequest($request);

        $formMenu = $this->createForm(SimpleMenuType::class, $menu);
        $formMenu->handleRequest($request);

        $defaultParams = array('layout' => null); // not break legacy
        $menuParams = (array) $menu->getParams();

        $menuParams = array_merge($defaultParams, $menuParams);

        $formMenuParams = $this->createForm(ListChildMenuType::class, $menuParams);
        $formMenuParams->handleRequest($request);

        if ($formContent->isSubmitted() && $formContent->isValid() && $formMenu->isSubmitted() && $formMenu->isValid() && $formMenuParams->isSubmitted() && $formMenuParams->isValid()) {
            //Force params to null for update | Bug Doctrine
            $menu->setParams(null);
            $this->em->persist($menu);
            $this->em->flush();

            // Update menu
            $menu->setSlug($slugify->slugify($menu->getTitle()));
            $params = (object) $formMenuParams->getData();
            $menu->setParams($params);
            $this->em->persist($menu);

            $this->em->persist($content);

            $this->em->flush();

            $this->addFlash('success', 'Page Liste enfant modifiée');

            return $this->redirectToRoute('admin_content_edit_list_child', array(
                'menu' => $menu->getId(),
                'content' => $content->getId(),
                'menu_type' => $menu->getMenuType(),
            ));
        }

        return $this->render('Admin/Content/editPage.html.twig', array(
            'form' => $formContent->createView(),
            'form_menu' => $formMenu->createView(),
            'form_menu_params' => $formMenuParams->createView(),
            'modules' => $modules,
            'menu' => $menu,
            'content' => $content,
        ));
    }

    /**
     *
     * @Route(
     *      "/menu/{menu_type}/add-content",
     *      name = "admin_content_add_content"
     * )
     *
     * @param Request $request
     * @param string menu_type
     * @return \Symfony\Component\HttpFoundation\RedirectResponse|\Symfony\Component\HttpFoundation\Response
     */
    public function addContentAction(Request $request, string $menu_type = 'main')
    {
        $contentId = $request->get('content');
        $menuId = $request->get('menu');

        $contentNew = new Content;
        $contentNew->setIsModule(false);
        $contentNew->setIsMain(false);
        $contentNew->setIsActu(false);

        // Formulaire page simple 
        $contentNew->setMenu($this->em->getRepository('AppBundle:Menu')->findOneById($menuId));
        $formContent = $this->createForm(BlocContentType::class, $contentNew, array(
            'action' => $this->generateUrl('admin_content_add_content', array(
                'content' => $contentId,
                'menu' => $menuId,
            ))
        ));
        $formContent->handleRequest($request);

        if ($formContent->isSubmitted() && $formContent->isValid()) {
            // On enregistre le contenu de la page simple 
            $this->em->persist($contentNew);
            $this->em->flush();

            $this->addFlash('success', 'Item ajouté');

            return $this->redirectToRoute('admin_content_edit_content', array(
                'menu' => $menuId,
                'content' => $contentNew->getId(),
                'contentbase' => $contentId,
            ));
        }

        return $this->render('Admin/Content/addContent.html.twig', array(
            'form' => $formContent->createView(),
        ));
    }

    /**
     *
     * @Route(
     *      "/menu/edit-content/{menu}/{content}",
     *      name = "admin_content_edit_content"
     * )
     *
     * @param Request $request
     * @param Menu $menu
     * @param Content $content
     * @param Content $contentbase
     * @return \Symfony\Component\HttpFoundation\RedirectResponse|\Symfony\Component\HttpFoundation\Response
     */
    public function editContentAction(Request $request, Menu $menu, Content $content, Content $contentbase)
    {
        // Formulaire page simple 
        $formContent = $this->createForm(BlocContentType::class, $content);
        $formContent->handleRequest($request);

        if ($formContent->isSubmitted() && $formContent->isValid()) {
            if ($request->get('remove_picture')) {
                $content->setPicture(null);
            }
            
            // On enregistre le contenu de la page simple
            $this->em->persist($content);
            $this->em->flush();

            $this->addFlash('success', 'Item modifié');

            return $this->redirectToRoute('admin_content_edit_content', array(
                'menu' => $menu->getId(),
                'content' => $content->getId(),
                'contentbase' => $contentbase->getId(),
            ));
        }

        return $this->render('Admin/Content/editContent.html.twig', array(
            'form' => $formContent->createView(),
            'menu' => $menu,
        ));
    }

    /**
     *
     * @Route(
     *      "/menu/add-module",
     *      name = "admin_content_add_module"
     * )
     *
     * @param Request $request
     * @param string menu_type
     * @return \Symfony\Component\HttpFoundation\RedirectResponse|\Symfony\Component\HttpFoundation\Response
     */
    public function addModuleAction(Request $request)
    {
        $contentId = $request->get('content');
        $menuId = $request->get('menu');

        $contentNew = new Content;
        $contentNew->setIsModule(true);
        $contentNew->setIsMain(false);
        $contentNew->setIsActu(false);

        // Formulaire page simple 
        $formContent = $this->createForm(ContentType::class, $contentNew, array(
            'action' => $this->generateUrl('admin_content_add_module', array(
                'content' => $contentId,
                'menu' => $menuId,
            ))
        ));
        $formContent->handleRequest($request);

        if ($formContent->isSubmitted() && $formContent->isValid()) {
            // On enregistre le contenu de la page simple 
            $contentNew->setMenu($this->em->getRepository('AppBundle:Menu')->findOneById($menuId));

            $this->em->persist($contentNew);
            $this->em->flush();

            $this->addFlash('success', 'Module ajouté');

            return $this->redirectToRoute('admin_content_edit_module', array(
                'menu' => $menuId,
                'content' => $contentNew->getId(),
                'contentbase' => $contentId,
            ));
        }

        return $this->render('Admin/Content/addModule.html.twig', array(
            'form' => $formContent->createView(),
        ));
    }

    /**
     *
     * @Route(
     *      "/menu/edit-module/{menu}/{content}/{contentbase}",
     *      name = "admin_content_edit_module"
     * )
     *
     * @param Request $request
     * @param Menu $menu
     * @param Content $content
     * @param Content $contentbase
     * @return \Symfony\Component\HttpFoundation\RedirectResponse|\Symfony\Component\HttpFoundation\Response
     */
    public function editModuleAction(Request $request, Menu $menu, Content $content, Content $contentbase)
    {
        // Formulaire page simple 
        $formContent = $this->createForm(ContentType::class, $content);
        $formContent->handleRequest($request);

        if ($formContent->isSubmitted() && $formContent->isValid()) {
            // On enregistre le contenu de la page simple 
            $this->em->persist($content);
            $this->em->flush();

            $this->addFlash('success', 'Module modifié');

            return $this->redirectToRoute('admin_content_edit_module',array(
                'menu' => $menu->getId(),
                'content' => $content->getId(),
                'contentbase' => $contentbase->getId(),
            ));
        }

        return $this->render('Admin/Content/editModule.html.twig', array(
            'form' => $formContent->createView(),
        ));
    }

    /**
     * Liste l'ensemble des articles
     *
     * @Route(
     *      "/actu/list/{type}/{page}",
     *      requirements = {"page": "\d+"},
     *      name = "admin_actu"
     * )
     *
     * @param Request $request
     * @param int $page
     * @return \Symfony\Component\HttpFoundation\Response
     */
    public function actuAction(Request $request, string $type = 'all', int $page = 1)
    {
        $resultsMaxPerPage = $this->getParameter('results.admin.max_per_page');
        $firstresult = ($page - 1) * $resultsMaxPerPage;

        // Mise à jour de la session avec les nouvelles valeurs
        $this->session->set($request->get('_route'), array(
            'page' => $page,
            'type' => $type,
        ));

        // On récupère toutes les catégories
        $querycategories = $this->em
            ->createQuery(
                'SELECT m FROM AppBundle:Menu m
                    WHERE m.typeTemplate = :type
                    ORDER BY m.position ASC, m.id ASC'
            )
            ->setParameter('type', $this->em->getRepository(TypeTemplate::class)->findOneByTitle(TypeTemplateTitle::NEWS));
        $categories = $querycategories->getResult();

        $querytemplate = $this->em
            ->createQuery(
                'SELECT s.id FROM AppBundle:TypeTemplate s
                    WHERE s.title = :type
                    ORDER BY s.id ASC'
            )
            ->setParameter('type', TypeTemplateTitle::NEWS);
        $template = $querytemplate->getResult();

        if ('all' == $type) {
            $templateid = $template[0]['id'];
            $dql = "SELECT m, n FROM AppBundle:Content m
                            LEFT JOIN m.menu n
                            WHERE m.is_actu = 't' AND n.id in (SELECT mm.id FROM AppBundle:Menu mm WHERE mm.typeTemplate = '$templateid')
                            ORDER BY  m.publish DESC,  m.created DESC";
        } else {
            $dql = "SELECT m FROM AppBundle:Content m
                            WHERE m.menu = '$type'
                            ORDER BY  m.publish DESC,  m.created DESC";
        }
        $query = $this->em
            ->createQuery($dql)
            ->setFirstResult($firstresult)
            ->setMaxResults($resultsMaxPerPage);

        $actus = new Paginator($query);

        $nbPage = count($actus) / $resultsMaxPerPage;

        // Constructeur pagination
        $pagination = array();
        for ($i = 1 ; $i < $nbPage + 1 ; $i++) {
            array_push($pagination, $i);
        }

        return $this->render('Admin/Actu/index.html.twig', array(
            'actus' => $actus,
            'type' => $type,
            'categories' => $categories,
            'pagination' => $pagination,
            'pageactuelle' => $page,
        ));
    }

    /**
     * Ajout d'un article
     *
     * @Route(
     *      "/actu/add/actu",
     *      name = "admin_actu_add"
     * )
     *
     * @param Request $request
     * @return \Symfony\Component\HttpFoundation\RedirectResponse|\Symfony\Component\HttpFoundation\Response
     */
    public function addActuAction(Request $request)
    {
        // Formulaire page simple 
        $content = new Content;
        $content->setIsModule(false);
        $content->setIsMain(true);
        $content->setIsActu(true);
        $formContent = $this->createForm(ActuContentType::class, $content);
        $formContent->handleRequest($request);

        if ($formContent->isSubmitted() && $formContent->isValid()) {
            // On enregistre le contenu de la page simple 
            $this->em->persist($content);
            $this->em->flush();

            $this->addFlash('success', 'Actu ajouté');

            return $this->redirectToRoute('admin_actu_edit', array(
                'id' => $content->getId(),
            ));
        }

        return $this->render('Admin/Actu/add.html.twig', array(
            'form' => $formContent->createView(),
        ));
    }

    /**
     *
     * @Route(
     *      "/actu/edit/{id}",
     *      requirements = {"id": "\d+"},
     *      name = "admin_actu_edit"
     * )
     *
     * @param Request $request
     * @param Content $content
     * @return \Symfony\Component\HttpFoundation\RedirectResponse|\Symfony\Component\HttpFoundation\Response
     */
    public function editActuAction(Request $request, Content $content)
    {
        $formContent = $this->createForm(ActuContentType::class, $content);
        $formContent->handleRequest($request);

        if ($formContent->isSubmitted() && $formContent->isValid()) {
            if ($request->get('remove_picture')) {
                $content->setPicture(null);
            }
            $content->setIsModule('f');
            $content->setIsMain('t');
            $content->setIsActu('t');
            $this->em->persist($content);

            $this->em->flush();

            $this->addFlash('success', 'Actu modifiée');

            return $this->redirectToRoute('admin_actu_edit', array(
                'id' => $content->getId(),
            ));
        }

        return $this->render('Admin/Actu/add.html.twig', array(
            'form' => $formContent->createView(),
        ));
    }

    
    /**
     *
     * @Route(
     *      "/menu/{menu_type}/add-category-media",
     *      name = "admin_content_add_category_media"
     * )
     *
     * @param Request $request
     * @param string menu_type
     * @return \Symfony\Component\HttpFoundation\RedirectResponse|Response
     */
    public function addCategoryMediaAction(Request $request, string $menu_type = 'main')
    {
        $slugify = new Slugify();

        // Formulaire page simple 
        $content = new Content;
        $content->setIsModule(false);
        $content->setIsMain(true);
        $content->setIsActu(false);
        $formContent = $this->createForm(SimpleContentType::class, $content);
        $formContent->handleRequest($request);

        // Formulaire menu article 
        $menu = new Menu;
        $menu->setMenuType($menu_type);
        $menu->setTypeTemplate($this->em->getRepository(TypeTemplate::class)->findOneByTitle(TypeTemplateTitle::MEDIA));
        $formMenu = $this->createForm(SimpleMenuType::class, $menu);
        $formMenu->handleRequest($request);

        if ($formContent->isSubmitted() && $formContent->isValid() && $formMenu->isSubmitted() && $formMenu->isValid()){
            // On enregistre le menu de la page simple 
            $menu->setSlug($slugify->slugify($menu->getTitle()));
            $this->em->persist($menu);

            // On enregistre le contenu de la page simple 
            $content->setMenu($menu);
            $this->em->persist($content);

            $this->em->flush();

            $this->addFlash('success', 'Menu ajouté');

            return $this->redirectToRoute('admin_content_edit_category_media', array(
                'menu' => $menu->getId(),
                'content' => $content->getId(),
                'menu_type' => $menu->getMenuType(),
            ));
        }

        return $this->render('Admin/Content/editPage.html.twig', array(
            'form' => $formContent->createView(),
            'form_menu' => $formMenu->createView(),
            'menu' => $menu,
        ));
    }

    /**
     *
     * @Route(
     *      "/menu/{menu_type}/edit-category-media/{menu}/{content}",
     *      name = "admin_content_edit_category_media"
     * )
     *
     * @param Request $request
     * @param Menu $menu
     * @param Content $content
     * @param string menu_type
     * @return \Symfony\Component\HttpFoundation\RedirectResponse|Response
     */
    public function editCategoryMediaAction(Request $request, Menu $menu, Content $content = null, string $menu_type = 'main')
    {
        if (is_null($content)) {
            $content = new Content;
            $content->setIsModule(false);
            $content->setIsMain(true);
            $content->setIsActu(false);
            $content->setMenu($menu);
        }

        $slugify = new Slugify();
        
        $formContent = $this->createForm(SimpleContentType::class, $content);
        $formContent->handleRequest($request);
        
        $formMenu = $this->createForm(SimpleMenuType::class, $menu);
        $formMenu->handleRequest($request);

        if ($formContent->isSubmitted() && $formContent->isValid() && $formMenu->isSubmitted() && $formMenu->isValid()){
            // On enregistre le menu de la page simple 
            $menu->setSlug($slugify->slugify($menu->getTitle()));
            $this->em->persist($menu);

            // On enregistre le contenu de la page simple 
            if ($request->get('remove_picture')) {
                $content->setPicture(null);
            }
            $this->em->persist($content);

            $this->em->flush();

            $this->addFlash('success', 'Menu modifié');

            return $this->redirectToRoute('admin_content_edit_category_media', array(
                'menu' => $menu->getId(),
                'content' => $content->getId(),
                'menu_type' => $menu->getMenuType(),
            ));
        }

        return $this->render('Admin/Content/editPage.html.twig', array(
            'form' => $formContent->createView(),
            'form_menu' => $formMenu->createView(),
            'menu' => $menu,
        ));
    }

    /**
     *
     * @Route(
     *      "/menu/{menu_type}/add-category-actu",
     *      name = "admin_content_add_category_actu"
     * )
     *
     * @param Request $request
     * @param string menu_type
     * @return \Symfony\Component\HttpFoundation\RedirectResponse|Response
     */
    public function addCategoryActuAction(Request $request, string $menu_type = 'main')
    {
        $slugify = new Slugify();
        
        // Formulaire page simple 
        $content = new Content;
        $content->setIsModule(false);
        $content->setIsMain(true);
        $content->setIsActu(false);
        $formContent = $this->createForm(SimpleContentType::class, $content);
        $formContent->handleRequest($request);

        // Formulaire menu article 
        $menu = new Menu;
        $menu->setMenuType($menu_type);
        $menu->setTypeTemplate($this->em->getRepository(TypeTemplate::class)->findOneByTitle(TypeTemplateTitle::NEWS));

        $formMenu = $this->createForm(SimpleMenuType::class, $menu);
        $formMenu->handleRequest($request);
        
        if ($formContent->isSubmitted() && $formContent->isValid() && $formMenu->isSubmitted() && $formMenu->isValid()){
            // On enregistre le menu de la page simple 
            $menu->setSlug($slugify->slugify($menu->getTitle()));
            $this->em->persist($menu);

            // On enregistre le contenu de la page simple 
            $content->setMenu($menu);
            $this->em->persist($content);

            $this->em->flush();

            $this->addFlash('success', 'Menu ajouté');

            return $this->redirectToRoute('admin_content_edit_category_actu', array(
                'menu' => $menu->getId(),
                'content' => $content->getId(),
                'menu_type' => $menu->getMenuType(),
            ));
        }

        return $this->render('Admin/Content/editPage.html.twig', array(
            'form' => $formContent->createView(),
            'form_menu' => $formMenu->createView(),
            'menu' => $menu,
        ));
    }

    /**
     *
     * @Route(
     *      "/menu/{menu_type}/edit-category-actu/{menu}/{content}",
     *      name = "admin_content_edit_category_actu"
     * )
     *
     * @param Request $request
     * @param Menu $menu
     * @param Content $content
     * @param string menu_type
     * @return \Symfony\Component\HttpFoundation\RedirectResponse|Response
     */
    public function editCategoryActuAction(Request $request, Menu $menu, Content $content = null, string $menu_type = 'main')
    {
        if (is_null($content)) {
            $content = new Content;
            $content->setIsModule(false);
            $content->setIsMain(true);
            $content->setIsActu(false);
            $content->setMenu($menu);
        }

        $slugify = new Slugify();

        $formContent = $this->createForm(SimpleContentType::class, $content);
        $formContent->handleRequest($request);
        
        $formMenu = $this->createForm(SimpleMenuType::class, $menu);
        $formMenu->handleRequest($request);

        if ($formContent->isSubmitted() && $formContent->isValid() && $formMenu->isSubmitted() && $formMenu->isValid()){
            // On enregistre le menu de la page simple 
            $menu->setSlug($slugify->slugify($menu->getTitle()));
            $this->em->persist($menu);

            // On enregistre le contenu de la page simple 
            if ($request->get('remove_picture')) {
                $content->setPicture(null);
            }
            $this->em->persist($content);

            $this->em->flush();

            $this->addFlash('success', 'Menu modifié');

            return $this->redirectToRoute('admin_content_edit_category_actu', array(
                'menu' => $menu->getId(),
                'content' => $content->getId(),
                'menu_type' => $menu->getMenuType(),
            ));
        }

        return $this->render('Admin/Content/editPage.html.twig', array(
            'form' => $formContent->createView(),
            'form_menu' => $formMenu->createView(),
            'menu' => $menu
        ));
    }

    /**
     * @Route(
     *      "/menu/{menu_type}/add-page-partners",
     *      name = "admin_content_add_page_partners"
     * )
     *
     * @param Request $request
     * @param string menu_type
     * @return \Symfony\Component\HttpFoundation\RedirectResponse|\Symfony\Component\HttpFoundation\Response
     */
    public function addPagePartnersAction(Request $request, string $menu_type = 'main')
    {
        $slugify = new Slugify();

        // Formulaire page simple 
        $content = new Content;
        $content->setIsModule(false);
        $content->setIsMain(true);
        $content->setIsActu(false);
        $formContent = $this->createForm(ContentType::class, $content);
        $formContent->handleRequest($request);

        // Formulaire menu article 
        $menu = new Menu;
        $menu->setMenuType($menu_type);
        $menu->setTypeTemplate($this->em->getRepository(TypeTemplate::class)->findOneByTitle(TypeTemplateTitle::PARTNERS));
        $formMenu = $this->createForm(SimpleMenuType::class, $menu);
        $formMenu->handleRequest($request);

        $formMenuParams = $this->createForm(PartnerMenuType::class, $menu->getParams());
        $formMenuParams->handleRequest($request);

        if ($formContent->isSubmitted() && $formContent->isValid() && $formMenu->isSubmitted() && $formMenu->isValid() && $formMenuParams->isSubmitted() && $formMenuParams->isValid()) {
            // On rengistre les paramètres du Menu
            $params = (object) $formMenuParams->getData();
            $menu->setParams($params);

            // On enregistre le menu de la page simple 
            $menu->setSlug($slugify->slugify($menu->getTitle()));
            $this->em->persist($menu);

            // On enregistre le contenu de la page simple 
            $content->setMenu($menu);
            $this->em->persist($content);

            $this->em->flush();

            $this->addFlash('success', 'Page Partenaires ajoutée');

            return $this->redirectToRoute('admin_content_edit_page_partners', array(
                'menu' => $menu->getId(),
                'content' => $content->getId(),
                'mode' => 'edit',
                'menu_type' => $menu->getMenuType(),
            ));
        }

        return $this->render('Admin/Content/editPage.html.twig', array(
            'form' => $formContent->createView(),
            'form_menu' => $formMenu->createView(),
            'form_menu_params' => $formMenuParams->createView(),
            'menu' => $menu,
        ));
    }


    /**
     *
     * @Route(
     *      "/menu/{menu_type}/edit-page-partners/{menu}/{content}",
     *      name = "admin_content_edit_page_partners"
     * )
     *
     * @param Request $request
     * @param Menu menu
     * @param Content $content
     * @param string menu_type
     * @return \Symfony\Component\HttpFoundation\RedirectResponse|\Symfony\Component\HttpFoundation\Response
     */
    public function editPagePartnersAction(Request $request, Menu $menu, Content $content, string $menu_type = 'main')
    {
        $slugify = new Slugify();

        /* $querymodules = $this->em->createQuery(
            "SELECT c FROM AppBundle:Content c WHERE c.menu=:c_menu and c.is_module=:c_is_module ORDER BY c.position ASC, c.id ASC"
        )->setParameters(array(
            'c_menu' => $menu->getId(),
            'c_is_module' => true,
        ));
        $modules = $querymodules->getResult();*/ 

        $modules = $this->em->getRepository(Module::class)->findByMenu(
            $menu->getId(), 
            array(
                //'isActive' => true,
                'notLikePosition' => array("home_%", "login_%", "modal_%", "menu_%", "social", "newsletter")
            )
        );

        $formContent = $this->createForm(ContentType::class, $content);
        $formContent->handleRequest($request);
        
        $formMenu = $this->createForm(SimpleMenuType::class, $menu);
        $formMenu->handleRequest($request);

        $defaultParams = array('filter_category' => null, 'filter_support' => null); // not break legacy
        $menuParams = (array) $menu->getParams();

        $menuParams = array_merge($defaultParams, $menuParams);

        $formMenuParams = $this->createForm(PartnerMenuType::class, $menuParams);
        $formMenuParams->handleRequest($request);

        if ($formContent->isSubmitted() && $formContent->isValid() && $formMenu->isSubmitted() && $formMenu->isValid() && $formMenuParams->isSubmitted() && $formMenuParams->isValid()) {
            //Force params to null for update | Bug Doctrine
            $menu->setParams(null);
            $this->em->persist($menu);
            $this->em->flush();

            // Update menu
            $menu->setSlug($slugify->slugify($menu->getTitle()));
            $params = (object) $formMenuParams->getData();
            $menu->setParams($params);

            $this->em->persist($menu);

            // Update content
            if ($request->get('remove_picture')) {
                $content->setPicture(null);
            }
            $this->em->persist($content);

            $this->em->flush();

            $this->addFlash('success', 'Page Partenaires modifiée');

            return $this->redirectToRoute('admin_content_edit_page_partners', array(
                'menu' => $menu->getId(),
                'content' => $content->getId(),
                'menu_type' => $menu->getMenuType(),
            ));
        }

        return $this->render('Admin/Content/editPage.html.twig', array(
            'form' => $formContent->createView(),
            'form_menu' => $formMenu->createView(),
            'form_menu_params' => $formMenuParams->createView(),
            'modules' => $modules,
            'menu' => $menu,
            'content' => $content,
        ));
    }

    /**
     * @Route(
     *      "/menu/{menu_type}/add-page-testimonials",
     *      name = "admin_content_add_page_testimonials"
     * )
     *
     * @param Request $request
     * @param string menu_type
     * @return \Symfony\Component\HttpFoundation\RedirectResponse|\Symfony\Component\HttpFoundation\Response
     */
    public function addPageTestimonialsAction(Request $request, string $menu_type = 'main')
    {
        $slugify = new Slugify();

        // Formulaire page simple 
        $content = new Content;
        $content->setIsModule(false);
        $content->setIsMain(true);
        $content->setIsActu(false);
        $formContent = $this->createForm(ContentType::class, $content);
        $formContent->handleRequest($request);

        // Formulaire menu article 
        $menu = new Menu;
        $menu->setMenuType($menu_type);
        $menu->setTypeTemplate($this->em->getRepository(TypeTemplate::class)->findOneByTitle(TypeTemplateTitle::TESTIMONIALS));
        $formMenu = $this->createForm(SimpleMenuType::class, $menu);
        $formMenu->handleRequest($request);

        if ($formContent->isSubmitted() && $formContent->isValid() && $formMenu->isSubmitted() && $formMenu->isValid()) {
            // On enregistre le menu de la page simple 
            $menu->setSlug($slugify->slugify($menu->getTitle()));
            $this->em->persist($menu);

            // On enregistre le contenu de la page simple 
            $content->setMenu($menu);
            $this->em->persist($content);

            $this->em->flush();

            $this->addFlash('success', 'Page Témoignages ajoutée');

            return $this->redirectToRoute('admin_content_edit_page_testimonials', array(
                'menu' => $menu->getId(),
                'content' => $content->getId(),
                'mode' => 'edit',
                'menu_type' => $menu->getMenuType(),
            ));
        }

        return $this->render('Admin/Content/editPage.html.twig', array(
            'form' => $formContent->createView(),
            'form_menu' => $formMenu->createView(),
            'menu' => $menu,
        ));
    }


    /**
     *
     * @Route(
     *      "/menu/{menu_type}/edit-page-testimonials/{menu}/{content}",
     *      name = "admin_content_edit_page_testimonials"
     * )
     *
     * @param Request $request
     * @param Menu menu
     * @param Content $content
     * @param string menu_type
     * @return \Symfony\Component\HttpFoundation\RedirectResponse|\Symfony\Component\HttpFoundation\Response
     */
    public function editPageTestimonialsAction(Request $request, Menu $menu, Content $content, string $menu_type = 'main')
    {
        $slugify = new Slugify();

        /* $querymodules = $this->em->createQuery(
            "SELECT c FROM AppBundle:Content c WHERE c.menu=:c_menu and c.is_module=:c_is_module ORDER BY c.position ASC, c.id ASC"
        )->setParameters(array(
            'c_menu' => $menu->getId(),
            'c_is_module' => true,
        ));
        $modules = $querymodules->getResult();*/ 

        $modules = $this->em->getRepository(Module::class)->findByMenu(
            $menu->getId(), 
            array(
                //'isActive' => true,
                'notLikePosition' => array("home_%", "login_%", "modal_%", "menu_%", "social", "newsletter")
            )
        );

        $formContent = $this->createForm(ContentType::class, $content);
        $formContent->handleRequest($request);
        
        $formMenu = $this->createForm(SimpleMenuType::class, $menu);
        $formMenu->handleRequest($request);

        if ($formContent->isSubmitted() && $formContent->isValid() && $formMenu->isSubmitted() && $formMenu->isValid()) {
            // Update menu
            $menu->setSlug($slugify->slugify($menu->getTitle()));
            $this->em->persist($menu);

            // Update content
            if ($request->get('remove_picture')) {
                $content->setPicture(null);
            }
            $this->em->persist($content);

            $this->em->flush();

            $this->addFlash('success', 'Page Témoignages modifiée');

            return $this->redirectToRoute('admin_content_edit_page_testimonials', array(
                'menu' => $menu->getId(),
                'content' => $content->getId(),
                'menu_type' => $menu->getMenuType(),
            ));
        }

        return $this->render('Admin/Content/editPage.html.twig', array(
            'form' => $formContent->createView(),
            'form_menu' => $formMenu->createView(),
            'modules' => $modules,
            'menu' => $menu,
            'content' => $content,
        ));
    }


    /**
     * @Route(
     *      "/menu/{menu_type}/add-page-contact",
     *      name = "admin_content_add_page_contact"
     * )
     *
     * @param Request $request
     * @param string menu_type
     * @return \Symfony\Component\HttpFoundation\RedirectResponse|\Symfony\Component\HttpFoundation\Response
     */
    public function addPageContactAction(Request $request, string $menu_type = 'main')
    {
        $slugify = new Slugify();

        // Formulaire page simple 
        $content = new Content;
        $content->setIsModule(false);
        $content->setIsMain(true);
        $content->setIsActu(false);
        $formContent = $this->createForm(ContentType::class, $content);
        $formContent->handleRequest($request);

        // Formulaire menu article 
        $menu = new Menu;
        $menu->setMenuType($menu_type);
        $menu->setTypeTemplate($this->em->getRepository(TypeTemplate::class)->findOneByTitle(TypeTemplateTitle::CONTACT));
        $formMenu = $this->createForm(SimpleMenuType::class, $menu);
        $formMenu->handleRequest($request);

        if ($formContent->isSubmitted() && $formContent->isValid() && $formMenu->isSubmitted() && $formMenu->isValid()) {
            // On enregistre le menu de la page simple 
            $menu->setSlug($slugify->slugify($menu->getTitle()));
            $this->em->persist($menu);

            // On enregistre le contenu de la page simple 
            $content->setMenu($menu);
            $this->em->persist($content);

            $this->em->flush();

            $this->addFlash('success', 'Page Contact ajouté');

            return $this->redirectToRoute('admin_content_edit_page_contact', array(
                'menu' => $menu->getId(),
                'content' => $content->getId(),
                'mode' => 'edit',
            ));
        }

        return $this->render('Admin/Content/editPage.html.twig', array(
            'form' => $formContent->createView(),
            'form_menu' => $formMenu->createView(),
            'menu' => $menu,
        ));
    }

    /**
     *
     * @Route(
     *      "/menu/{menu_type}/edit-page-contact/{menu}/{content}",
     *      name = "admin_content_edit_page_contact"
     * )
     *
     * @param Request $request
     * @param Menu menu
     * @param Content $content
     * @param string menu_type
     * @return \Symfony\Component\HttpFoundation\RedirectResponse|\Symfony\Component\HttpFoundation\Response
     */
    public function editPageContactAction(Request $request, Menu $menu, Content $content, string $menu_type = 'main')
    {
        $slugify = new Slugify();

        /* $querymodules = $this->em->createQuery(
            "SELECT c FROM AppBundle:Content c WHERE c.menu=:c_menu and c.is_module=:c_is_module ORDER BY c.position ASC, c.id ASC"
        )->setParameters(array(
            'c_menu' => $menu->getId(),
            'c_is_module' => true,
        ));
        $modules = $querymodules->getResult();*/ 

        $modules = $this->em->getRepository(Module::class)->findByMenu(
            $menu->getId(), 
            array(
                //'isActive' => true,
                'notLikePosition' => array("home_%", "login_%", "modal_%", "menu_%", "social", "newsletter")
            )
        );

        $formContent = $this->createForm(ContentType::class, $content);
        $formContent->handleRequest($request);
        
        $formMenu = $this->createForm(SimpleMenuType::class, $menu);
        $formMenu->handleRequest($request);

        if ($formContent->isSubmitted() && $formContent->isValid() && $formMenu->isSubmitted() && $formMenu->isValid()) {
            // Update menu
            $menu->setSlug($slugify->slugify($menu->getTitle()));
            $this->em->persist($menu);

            // Update content
            if ($request->get('remove_picture')) {
                $content->setPicture(null);
            }
            $this->em->persist($content);

            $this->em->flush();

            $this->addFlash('success', 'Page contact modifiée');

            return $this->redirectToRoute('admin_content_edit_page_contact', array(
                'menu' => $menu->getId(),
                'content' => $content->getId(),
                'menu_type' => $menu->getMenuType(),
            ));
        }

        return $this->render('Admin/Content/editPage.html.twig', array(
            'form' => $formContent->createView(),
            'form_menu' => $formMenu->createView(),
            'modules' => $modules,
            'menu' => $menu,
            'content' => $content,
        ));
    }
}
