<?php

namespace AppBundle\Pdf;

use AppBundle\Entity\DemandePEE;
use AppBundle\Entity\OperationPEE;
use AppBundle\Enum\OperationPEEState;
use Knp\Snappy\Pdf;
use Symfony\Component\HttpFoundation\Response;
use Symfony\Component\Security\Core\Exception\AccessDeniedException;
use Symfony\Component\Templating\EngineInterface;

/**
 * 
 * @author Bastien Gatellier <contact@bgatellier.fr>
 */
class PdfGenerator
{
    const FORMATS_ALLOWED = ['pdf', 'html'];
    const OPERATION_STATES_FORBIDDEN = [
        OperationPEEState::CANCELLED,
        OperationPEEState::CLOSED,
    ];

    /**
     * @var Pdf
     */
    private $generator;

    /**
     * @var EngineInterface
     */
    private $templating;

    /**
     * Constructor
     * @param Pdf $generator  
     * @param EngineInterface   $templating 
     */
    public function __construct(Pdf $generator, EngineInterface $templating)
    {
        $this->generator = $generator;
        $this->templating = $templating;
    }

    /**
     *
     * @param  OperationPEE $o The operation must nt be cancelled of closed.
     * @param  string $format Allowed formats: 'html' and 'pdf'. Default 'pdf'.
     *
     * @return Response
     */
    public function generateOperationPee(OperationPEE $o, string $format = 'pdf'): Response
    {
        $this->checkFormat($format);
        $this->checkOperationState($o->getState());

        // Generate the template that will be used for PDF generation
        $view = $this->templating->render('Admin/Ope/PEE/printOperation.html.twig', ['operation' => $o]);

        $responseContent = $view;
        $responseHeaders = [];

        if ('pdf' === $format) {
            $responseContent = $this->generator->getOutputFromHtml($view, [
                'dpi' => 96,
                'image-quality' => 100,
                'margin-top' => 0,
                'margin-right' => 0,
                'margin-bottom' => 0,
                'margin-left' => 0,
            ]);

            $responseHeaders = [
                'Content-Type'          => 'application/pdf',
                'Content-Disposition'   => 'attachment; filename="' . $o->getPdfReadableName() . '"'
            ];
        }

        return new Response($responseContent, Response::HTTP_OK, $responseHeaders);
    }

    /**
     * 
     * @param  string $format
     * @throws \InvalidArgumentException
     */
    private function checkFormat(string $format): void
    {
        if (!in_array($format, self::FORMATS_ALLOWED)) {
            throw new \InvalidArgumentException();
        }
    }

    /**
     * 
     * @param  string $state
     * @throws AccessDeniedException
     */
    private function checkOperationState(string $state): void
    {
        if (in_array($state, self::OPERATION_STATES_FORBIDDEN)) {
            throw new AccessDeniedException();
        }
    }

    /**
     * @param DemandePEE $d
     * @param string|string $format
     * @param string|string $numerDispoCEFPEP
     */
    public function generateCefpep(DemandePEE $d, string $format = 'pdf', string $numerDispoCEFPEP = null): Response
    {
        $this->checkFormat($format);

      
        // Generate the template that will be used for PDF generation
        $view = $this->templating->render('/Admin/Ope/PEE/printCefpep.html.twig', [
            'demande' => $d, 
            'numerDispoCEFPEP' => $numerDispoCEFPEP
            ]);

        $responseContent = $this->generator->getOutputFromHtml($view, [
//                'dpi' => 96,
                'image-quality' => 100,
                'margin-top' => 0,
                'margin-right' => 0,
                'margin-bottom' => 0,
                'margin-left' => 0,
            ], true);
//
        $responseHeaders = [
            'Content-Type'          => 'application/pdf',
            'Content-Disposition'   => 'filename="test.pdf"'
        ];

        return new Response($responseContent, Response::HTTP_OK, $responseHeaders);
    }
}
