<?php

namespace AppBundle\Form\Type;

use AppBundle\Entity\Media;
use AppBundle\Entity\Menu;
use AppBundle\Enum\MediaType as MediaTypeEnum;
use AppBundle\Enum\TypeTemplateTitle;
use AppBundle\Translation\ConstTranslator;
use Doctrine\ORM\EntityRepository;
use Symfony\Bridge\Doctrine\Form\Type\EntityType;
use Symfony\Component\Form\AbstractType;
use Symfony\Component\Form\Extension\Core\Type\CheckboxType;
use Symfony\Component\Form\Extension\Core\Type\ChoiceType;
use Symfony\Component\Form\Extension\Core\Type\DateType;
use Symfony\Component\Form\Extension\Core\Type\FileType;
use Symfony\Component\Form\Extension\Core\Type\HiddenType;
use Symfony\Component\Form\Extension\Core\Type\NumberType;
use Symfony\Component\Form\Extension\Core\Type\TextType;
use Symfony\Component\Form\Extension\Core\Type\TextareaType;
use Symfony\Component\Form\FormBuilderInterface;
use Symfony\Component\OptionsResolver\OptionsResolver;
use AppBundle\Enum\UserType;

class MediaType extends AbstractType
{
    private $userTypes;

    /**
     * Constructor
     * 
     * @param ConstTranslator $utils
     */
    public function __construct(ConstTranslator $utils)
    {
        $this->userTypes = $utils->trans(UserType::class);
    }

    /**
     * 
     * @param FormBuilderInterface $builder 
     * @param array $options 
     */
    public function buildForm(FormBuilderInterface $builder, array $options)
    {
        $type = $options['data']->getType();

        $builder
            ->add('title', TextType::class, array(
                'label' => 'Titre',
            ))
            ->add('linkLabel', TextType::class, array(
                'label' => 'Libellé du lien',
                'required' => false
            ))
            ->add('description', TextareaType::class, array(
                'attr' => array(
                    'class' => 'tinymce',
                ),
                'label' => 'Description',
                'required' => false
            ));

        switch ($type) {
            case MediaTypeEnum::VIDEO:
                $builder
                    ->add('path', TextType::class, array(
                        'label' => 'ID YouTube',
                        'help' => 'ID uniquement, pas URL. <br/>Exemple : https://www.youtube.com/watch?v=<strong>4yhJRSu3XzY</strong> devient <strong>4yhJRSu3XzY</strong>'
                    ))
                    ->add('pagePerso', ChoiceType::class, array(
                        'choices' => array_flip($this->userTypes),
                        'expanded' => true,
                        'label' => 'Emplacement espace personnel',
                        'multiple' => true,
                        'required'  => false,
                    ))
                    ->add('opeItdlcEmail', ChoiceType::class, array(
                        'choices' => array_flip($this->userTypes),
                        'expanded' => true,
                        'label' => 'Email opération ITDLC',
                        'multiple' => true,
                    ))
                    ->add('menu', EntityType::class, array(
                        'class' => Menu::class,
                        'expanded'  => true,
                        'label' => 'Emplacement page de présentation',
                        'multiple'  => true,
                        'choice_label'  => function($menu, $key, $index) {
                            return $menu->getTitle() . ' (' . $menu->getTypetemplate()->getTitle() . ')';
                        },
                        'query_builder' => function(EntityRepository $er) {
                            return $er->createQueryBuilder('m')
                                ->leftJoin('m.typeTemplate', 'tt')
                                ->where('tt.title = :tt_title1')
                                ->orWhere('tt.title = :tt_title2')
                                ->orderBy('m.title', 'ASC')
                                ->setParameters(array(
                                    'tt_title1' => TypeTemplateTitle::PRESENTATION,
                                    'tt_title2' => TypeTemplateTitle::MEDIA,
                                ));
                        },
                        'required'  => false,
                    ))
                    ->add('created', DateType::class, array(
                        'attr' => array(
                            'class' => 'date',
                        ),
                        'format' => 'dd/MM/yyyy',
                        'label' => 'Date de création',
                        'required' => false,
                        'widget' => 'single_text',
                    ));
                break;

            case MediaTypeEnum::PICTURE:
                $builder
                    ->add('filePicture', FileType::class, array(
                        'attr'=>array(
                            'class' => 'image_input',
                        ),
                        'help' => 'Dimensions maximales:  1728x972',
                        'image_path' => 'webPicture',
                        'label' => 'Image',
                        'required' => false,
                    ))
                    ->add('menu', EntityType::class, array(
                        'class' => Menu::class,
                        'expanded'  => true,
                        'label' => 'Emplacement page de présentation',
                        'multiple'  => true,
                        'choice_label'  => 'title',
                        'query_builder' => function(EntityRepository $er) {
                            return $er->createQueryBuilder('m')
                                ->leftJoin('m.typeTemplate', 'tt')
                                ->where('tt.title = :tt_title')
                                ->orderBy('m.position', 'ASC')
                                ->setParameter('tt_title', TypeTemplateTitle::PRESENTATION);
                        },
                        'required'  => false,
                    ));
                break;
            
            default: // MediaTypeEnum::PDF
                $builder
                    ->add('filePdf', FileType::class, array(
                        'attr'=>array(
                            'class' => 'pdf_input',
                        ),
                        'image_path' => 'webPdf',
                        'label' => 'Pdf',
                        'required' => false,
                    ))
                    ->add('pagePerso', ChoiceType::class, array(
                        'choices' => array_flip($this->userTypes),
                        'expanded' => true,
                        'label' => 'Emplacement espace personnel',
                        'multiple' => true,
                    ))
                    ->add('opeItdlcEmail', ChoiceType::class, array(
                        'choices' => array_flip($this->userTypes),
                        'expanded' => true,
                        'label' => 'Email opération ITDLC',
                        'multiple' => true,
                    ));
                break;
        }

        $builder
            ->add('position', NumberType::class, array(
                'label' => 'Position',
                'required' => false,
            ))
            ->add('is_active', CheckboxType::class, array(
                'label' => 'Actif',
                'required' => false,
            ));
    }

    /**
     * 
     * @param OptionsResolver $resolver 
     */
    public function configureOptions(OptionsResolver $resolver)
    {
        $resolver->setDefaults(array(
            'data_class' => Media::class,
            'translation_domain' => 'form',
        ));
    }
}
