<?php

namespace AppBundle\Helper;

use AppBundle\Entity\OperationPEE;
use AppBundle\Entity\SMS;
use AppBundle\Entity\SMSCampaign;
use AppBundle\Entity\SMSCampaignPEE;
use AppBundle\Entity\User;
use AppBundle\SMS\SMSSender;
use Doctrine\Common\Collections\ArrayCollection;

class SMSHelper
{
    /**
     * @var SMSCampaign
     */
    private static $campaign;
    private static $sender;

    /**
     * Init Sender
     */
    public static function initSender(SMSSender $sender)
    {
        self::$sender = $sender;
    }

    /**
     * Init Campaign
     * 
     * @param array $data
     * 
     * @return SMSCampaignPEE|array
     */
    public static function initCampaignPEE(array $data)
    {        
        $ope = (isset($data['content'])) ? $data['ope'] : null;
        $content = (isset($data['content'])) ? $data['content'] : null;

        if ( empty($ope) || empty($content) ) {
            return array (
                'result' => false,
                'error' => 'Empty Params (ope or content)',        
            );
        }
        
        self::$campaign = new SMSCampaignPEE;
        self::$campaign->setOperationpee($ope);
        self::$campaign->setContent(self::prepareCampaignPEEContent($content));

        return self::$campaign;
    }

    /**
     * reset campaign
     * 
     * @return SMSCampaignPEE
     */
    public static function resetCampaign()
    {
        $campaign = self::$campaign;

        self::$campaign = null;
        
        return $campaign;
    }

    /**
     * get campaign updated
     * 
     * @return SMSCampaignPEE
     */
    public static function getCampaign()
    {
        return self::$campaign;
    }

    /**
     * Manipulate Content for SMSCampaign Entity
     * 
     * @param string $content
     * @param OperationPEE $ope
     * 
     * @return string
     */
    public static function prepareCampaignPEEContent(string $content, OperationPEE $ope = null)
    {
        if (empty($ope) || !$ope instanceof OperationPEE) {
            $ope = self::$campaign->getOperationpee();
        }

        $opeDateVisite = $ope->getDateVisite();
        $opeStartTime = $ope->getStartTime();
        $opeEndTime = $ope->getEndTime();
                
        $replacements = array(
            '__OPE_COMPANY__' => $ope->getCompany(),
            '__OPE_DATE__' => ($opeDateVisite instanceof \DateTime) ? $opeDateVisite->format('d/m/Y') : "(date inconnue)",
            '__OPE_TIME_START__' => ($opeStartTime instanceof \DateTime) ? $opeStartTime->format('H\hi') : "(horaire inconnu)",
            '__OPE_TIME_END__' => ($opeEndTime instanceof \DateTime) ? $opeEndTime->format('H\hi') : "(horaire inconnu)",
            '__OPE_FULL_ADDRESS__' => $ope->getSite() . ' ,' . $ope->getAddress() . ' ,' . $ope->getZipcode() . ' ' . $ope->getCity(),
            '__OPE_SITE__' => $ope->getSite(),
            '__OPE_ADDRESS__' => $ope->getAddress(),
            '__OPE_CITY__' => $ope->getCity(),
            '__OPE_ZIPCODE__' => $ope->getZipcode(),
            '__OPE_ACADEMY__' => $ope->getAcademy(),
        ); 
        
        $content = str_replace(array_keys($replacements), array_values($replacements), $content);

        // Remove all accents and special characters
        //return self::slugifyText($content);
        return $content;
    }
    
    /**
     * Prepare SMS Entities
     * 
     * @param array $data
     * 
     * @return ArrayCollection|array
     */
    public static function prepareSMSList(array $data)
    {
        //Get Recipients
        $recipients = (isset($data['recipients'])) ? $data['recipients'] : array();

        if ( empty($recipients) ) {
            return array (
                'result' => false,
                'error' => 'Empty Params (recipients)',        
            );
        }

        foreach($recipients as $demandepee) {
            $recipient = $demandepee->getCuser();

            //Prepare Phone
            $mobile_recipient = $recipient->getMobile();
            $replacements = array(" ", "-", "_", "(", ")", "/");
            $mobile_recipient = str_replace($replacements, "", $mobile_recipient);

            if (!preg_match('/^(06|07)[0-9]{8}$/', $mobile_recipient)) {
                $mobile_recipient = $recipient->getPhone();
                $replacements = array(" ", "-", "_", "(", ")", "/"); 
                $mobile_recipient = str_replace($replacements, "", $mobile_recipient);

                if (!preg_match('/^(06|07)[0-9]{8}$/', $mobile_recipient)) {
                    continue;  
                }   
            }
            $mobile_recipient = preg_replace("/^0([0-9]{1})([0-9]{8})$/", "0033$1$2", $mobile_recipient, 1);

            $sms = new SMS;
            $sms->setCampaign(self::$campaign);
            $sms->setUser($recipient);
            $sms->setMobile($mobile_recipient);
            $sms->setContent(self::prepareSMSContent($recipient));

            self::$campaign->addChildren($sms);
        }
        
        return self::$campaign->getChildren();
    }


    /**
     * Manipulate Content for SMS Entity
     * 
     * @param User $recipient
     *  
     * @return string
     */
    private static function prepareSMSContent(User $recipient)
    {
        $content = self::$campaign->getContent();
                
        $replacements = array(
            '__USER_FULLNAME__' => $recipient->getFirstname() . ' ' . $recipient->getLastname(),
            '__USER_FIRSTNAME__' => $recipient->getFirstname(),
            '__USER_LASTNAME__' => $recipient->getLastname(),
            '__USER_SHOES__' => $recipient->getShoeSize(),
        );
        
        $content = str_replace(array_keys($replacements), array_values($replacements), $content);

        // Remove all accents and special characters
        //return self::slugifyText($content);
        return $content;
    }

    /**
     * Send SMS Entities with Sendin Blue API
     * 
     * @param array $data
     * 
     * @return ArrayCollection|array
     */
    public static function sendSMSList()
    {
        if ( empty(self::$sender) ) {
            return array (
                'result' => false,
                'error' => 'Sender not initialized',        
            );
        }

        if ( empty(self::$campaign) ) {
            return array (
                'result' => false,
                'error' => 'Campaign not initialized',        
            );
        }
        
        $ope = self::$campaign->getOperationpee();

        if ( empty($ope) ) {
            return array (
                'result' => false,
                'error' => 'Operation not setted',        
            );
        }

        $tag = "Operation PEE " . $ope->getCompanyText() . " " . $ope->getSite() . " " . ( ($ope->getDateVisite()) ? ' du ' . $ope->getDateVisite()->format("d-m-Y") : '' );
        // Remove all accents and special characters
        $tag = self::slugifyText($tag);

        $smsList = self::$campaign->getChildren();

        self::$campaign->setSendedDateStart(new \DateTime());
        foreach($smsList as &$sms) {
            self::$sender->sendSMS($sms, false, $tag);
        }
        $hasErrors = count(self::$sender->getErrors()) > 0;
        self::$campaign->setSendedDateEnd(new \DateTime());

        if ($hasErrors) {
            self::$campaign->setSended(SMSCampaign::STATE_ERROR);
        }
        else {
            self::$campaign->setSended(SMSCampaign::STATE_SENDED);
        }

        return self::$sender->getResponses();
    }

    /**
     * Remove all accents and special characters
     */
    private static function slugifyText($text)
    {
        // Convert accents and special characters
        //$replacements = array('/' => '-', "\n" => ' ', "\r" => ' ', 'á' => 'a', 'Á' => 'A', 'à' => 'a', 'À' => 'A', 'ă' => 'a', 'Ă' => 'A', 'â' => 'a', 'Â' => 'A', 'å' => 'a', 'Å' => 'A', 'ã' => 'a', 'Ã' => 'A', 'ą' => 'a', 'Ą' => 'A', 'ā' => 'a', 'Ā' => 'A', 'ä' => 'ae', 'Ä' => 'AE', 'æ' => 'ae', 'Æ' => 'AE', 'ḃ' => 'b', 'Ḃ' => 'B', 'ć' => 'c', 'Ć' => 'C', 'ĉ' => 'c', 'Ĉ' => 'C', 'č' => 'c', 'Č' => 'C', 'ċ' => 'c', 'Ċ' => 'C', 'ç' => 'c', 'Ç' => 'C', 'ď' => 'd', 'Ď' => 'D', 'ḋ' => 'd', 'Ḋ' => 'D', 'đ' => 'd', 'Đ' => 'D', 'ð' => 'dh', 'Ð' => 'Dh', 'é' => 'e', 'É' => 'E', 'è' => 'e', 'È' => 'E', 'ĕ' => 'e', 'Ĕ' => 'E', 'ê' => 'e', 'Ê' => 'E', 'ě' => 'e', 'Ě' => 'E', 'ë' => 'e', 'Ë' => 'E', 'ė' => 'e', 'Ė' => 'E', 'ę' => 'e', 'Ę' => 'E', 'ē' => 'e', 'Ē' => 'E', 'ḟ' => 'f', 'Ḟ' => 'F', 'ƒ' => 'f', 'Ƒ' => 'F', 'ğ' => 'g', 'Ğ' => 'G', 'ĝ' => 'g', 'Ĝ' => 'G', 'ġ' => 'g', 'Ġ' => 'G', 'ģ' => 'g', 'Ģ' => 'G', 'ĥ' => 'h', 'Ĥ' => 'H', 'ħ' => 'h', 'Ħ' => 'H', 'í' => 'i', 'Í' => 'I', 'ì' => 'i', 'Ì' => 'I', 'î' => 'i', 'Î' => 'I', 'ï' => 'i', 'Ï' => 'I', 'ĩ' => 'i', 'Ĩ' => 'I', 'į' => 'i', 'Į' => 'I', 'ī' => 'i', 'Ī' => 'I', 'ĵ' => 'j', 'Ĵ' => 'J', 'ķ' => 'k', 'Ķ' => 'K', 'ĺ' => 'l', 'Ĺ' => 'L', 'ľ' => 'l', 'Ľ' => 'L', 'ļ' => 'l', 'Ļ' => 'L', 'ł' => 'l', 'Ł' => 'L', 'ṁ' => 'm', 'Ṁ' => 'M', 'ń' => 'n', 'Ń' => 'N', 'ň' => 'n', 'Ň' => 'N', 'ñ' => 'n', 'Ñ' => 'N', 'ņ' => 'n', 'Ņ' => 'N', 'ó' => 'o', 'Ó' => 'O', 'ò' => 'o', 'Ò' => 'O', 'ô' => 'o', 'Ô' => 'O', 'ő' => 'o', 'Ő' => 'O', 'õ' => 'o', 'Õ' => 'O', 'ø' => 'oe', 'Ø' => 'OE', 'ō' => 'o', 'Ō' => 'O', 'ơ' => 'o', 'Ơ' => 'O', 'ö' => 'oe', 'Ö' => 'OE', 'ṗ' => 'p', 'Ṗ' => 'P', 'ŕ' => 'r', 'Ŕ' => 'R', 'ř' => 'r', 'Ř' => 'R', 'ŗ' => 'r', 'Ŗ' => 'R', 'ś' => 's', 'Ś' => 'S', 'ŝ' => 's', 'Ŝ' => 'S', 'š' => 's', 'Š' => 'S', 'ṡ' => 's', 'Ṡ' => 'S', 'ş' => 's', 'Ş' => 'S', 'ș' => 's', 'Ș' => 'S', 'ß' => 'SS', 'ť' => 't', 'Ť' => 'T', 'ṫ' => 't', 'Ṫ' => 'T', 'ţ' => 't', 'Ţ' => 'T', 'ț' => 't', 'Ț' => 'T', 'ŧ' => 't', 'Ŧ' => 'T', 'ú' => 'u', 'Ú' => 'U', 'ù' => 'u', 'Ù' => 'U', 'ŭ' => 'u', 'Ŭ' => 'U', 'û' => 'u', 'Û' => 'U', 'ů' => 'u', 'Ů' => 'U', 'ű' => 'u', 'Ű' => 'U', 'ũ' => 'u', 'Ũ' => 'U', 'ų' => 'u', 'Ų' => 'U', 'ū' => 'u', 'Ū' => 'U', 'ư' => 'u', 'Ư' => 'U', 'ü' => 'ue', 'Ü' => 'UE', 'ẃ' => 'w', 'Ẃ' => 'W', 'ẁ' => 'w', 'Ẁ' => 'W', 'ŵ' => 'w', 'Ŵ' => 'W', 'ẅ' => 'w', 'Ẅ' => 'W', 'ý' => 'y', 'Ý' => 'Y', 'ỳ' => 'y', 'Ỳ' => 'Y', 'ŷ' => 'y', 'Ŷ' => 'Y', 'ÿ' => 'y', 'Ÿ' => 'Y', 'ź' => 'z', 'Ź' => 'Z', 'ž' => 'z', 'Ž' => 'Z', 'ż' => 'z', 'Ż' => 'Z', 'þ' => 'th', 'Þ' => 'Th', 'µ' => 'u', 'а' => 'a', 'А' => 'a', 'б' => 'b', 'Б' => 'b', 'в' => 'v', 'В' => 'v', 'г' => 'g', 'Г' => 'g', 'д' => 'd', 'Д' => 'd', 'е' => 'e', 'Е' => 'E', 'ё' => 'e', 'Ё' => 'E', 'ж' => 'zh', 'Ж' => 'zh', 'з' => 'z', 'З' => 'z', 'и' => 'i', 'И' => 'i', 'й' => 'j', 'Й' => 'j', 'к' => 'k', 'К' => 'k', 'л' => 'l', 'Л' => 'l', 'м' => 'm', 'М' => 'm', 'н' => 'n', 'Н' => 'n', 'о' => 'o', 'О' => 'o', 'п' => 'p', 'П' => 'p', 'р' => 'r', 'Р' => 'r', 'с' => 's', 'С' => 's', 'т' => 't', 'Т' => 't', 'у' => 'u', 'У' => 'u', 'ф' => 'f', 'Ф' => 'f', 'х' => 'h', 'Х' => 'h', 'ц' => 'c', 'Ц' => 'c', 'ч' => 'ch', 'Ч' => 'ch', 'ш' => 'sh', 'Ш' => 'sh', 'щ' => 'sch', 'Щ' => 'sch', 'ъ' => '', 'Ъ' => '', 'ы' => 'y', 'Ы' => 'y', 'ь' => '', 'Ь' => '', 'э' => 'e', 'Э' => 'e', 'ю' => 'ju', 'Ю' => 'ju', 'я' => 'ja', 'Я' => 'ja');
        $replacements = array('/' => '-', 'á' => 'a', 'Á' => 'A', 'à' => 'a', 'À' => 'A', 'ă' => 'a', 'Ă' => 'A', 'â' => 'a', 'Â' => 'A', 'å' => 'a', 'Å' => 'A', 'ã' => 'a', 'Ã' => 'A', 'ą' => 'a', 'Ą' => 'A', 'ā' => 'a', 'Ā' => 'A', 'ä' => 'ae', 'Ä' => 'AE', 'æ' => 'ae', 'Æ' => 'AE', 'ḃ' => 'b', 'Ḃ' => 'B', 'ć' => 'c', 'Ć' => 'C', 'ĉ' => 'c', 'Ĉ' => 'C', 'č' => 'c', 'Č' => 'C', 'ċ' => 'c', 'Ċ' => 'C', 'ç' => 'c', 'Ç' => 'C', 'ď' => 'd', 'Ď' => 'D', 'ḋ' => 'd', 'Ḋ' => 'D', 'đ' => 'd', 'Đ' => 'D', 'ð' => 'dh', 'Ð' => 'Dh', 'é' => 'e', 'É' => 'E', 'è' => 'e', 'È' => 'E', 'ĕ' => 'e', 'Ĕ' => 'E', 'ê' => 'e', 'Ê' => 'E', 'ě' => 'e', 'Ě' => 'E', 'ë' => 'e', 'Ë' => 'E', 'ė' => 'e', 'Ė' => 'E', 'ę' => 'e', 'Ę' => 'E', 'ē' => 'e', 'Ē' => 'E', 'ḟ' => 'f', 'Ḟ' => 'F', 'ƒ' => 'f', 'Ƒ' => 'F', 'ğ' => 'g', 'Ğ' => 'G', 'ĝ' => 'g', 'Ĝ' => 'G', 'ġ' => 'g', 'Ġ' => 'G', 'ģ' => 'g', 'Ģ' => 'G', 'ĥ' => 'h', 'Ĥ' => 'H', 'ħ' => 'h', 'Ħ' => 'H', 'í' => 'i', 'Í' => 'I', 'ì' => 'i', 'Ì' => 'I', 'î' => 'i', 'Î' => 'I', 'ï' => 'i', 'Ï' => 'I', 'ĩ' => 'i', 'Ĩ' => 'I', 'į' => 'i', 'Į' => 'I', 'ī' => 'i', 'Ī' => 'I', 'ĵ' => 'j', 'Ĵ' => 'J', 'ķ' => 'k', 'Ķ' => 'K', 'ĺ' => 'l', 'Ĺ' => 'L', 'ľ' => 'l', 'Ľ' => 'L', 'ļ' => 'l', 'Ļ' => 'L', 'ł' => 'l', 'Ł' => 'L', 'ṁ' => 'm', 'Ṁ' => 'M', 'ń' => 'n', 'Ń' => 'N', 'ň' => 'n', 'Ň' => 'N', 'ñ' => 'n', 'Ñ' => 'N', 'ņ' => 'n', 'Ņ' => 'N', 'ó' => 'o', 'Ó' => 'O', 'ò' => 'o', 'Ò' => 'O', 'ô' => 'o', 'Ô' => 'O', 'ő' => 'o', 'Ő' => 'O', 'õ' => 'o', 'Õ' => 'O', 'ø' => 'oe', 'Ø' => 'OE', 'ō' => 'o', 'Ō' => 'O', 'ơ' => 'o', 'Ơ' => 'O', 'ö' => 'oe', 'Ö' => 'OE', 'ṗ' => 'p', 'Ṗ' => 'P', 'ŕ' => 'r', 'Ŕ' => 'R', 'ř' => 'r', 'Ř' => 'R', 'ŗ' => 'r', 'Ŗ' => 'R', 'ś' => 's', 'Ś' => 'S', 'ŝ' => 's', 'Ŝ' => 'S', 'š' => 's', 'Š' => 'S', 'ṡ' => 's', 'Ṡ' => 'S', 'ş' => 's', 'Ş' => 'S', 'ș' => 's', 'Ș' => 'S', 'ß' => 'SS', 'ť' => 't', 'Ť' => 'T', 'ṫ' => 't', 'Ṫ' => 'T', 'ţ' => 't', 'Ţ' => 'T', 'ț' => 't', 'Ț' => 'T', 'ŧ' => 't', 'Ŧ' => 'T', 'ú' => 'u', 'Ú' => 'U', 'ù' => 'u', 'Ù' => 'U', 'ŭ' => 'u', 'Ŭ' => 'U', 'û' => 'u', 'Û' => 'U', 'ů' => 'u', 'Ů' => 'U', 'ű' => 'u', 'Ű' => 'U', 'ũ' => 'u', 'Ũ' => 'U', 'ų' => 'u', 'Ų' => 'U', 'ū' => 'u', 'Ū' => 'U', 'ư' => 'u', 'Ư' => 'U', 'ü' => 'ue', 'Ü' => 'UE', 'ẃ' => 'w', 'Ẃ' => 'W', 'ẁ' => 'w', 'Ẁ' => 'W', 'ŵ' => 'w', 'Ŵ' => 'W', 'ẅ' => 'w', 'Ẅ' => 'W', 'ý' => 'y', 'Ý' => 'Y', 'ỳ' => 'y', 'Ỳ' => 'Y', 'ŷ' => 'y', 'Ŷ' => 'Y', 'ÿ' => 'y', 'Ÿ' => 'Y', 'ź' => 'z', 'Ź' => 'Z', 'ž' => 'z', 'Ž' => 'Z', 'ż' => 'z', 'Ż' => 'Z', 'þ' => 'th', 'Þ' => 'Th', 'µ' => 'u', 'а' => 'a', 'А' => 'a', 'б' => 'b', 'Б' => 'b', 'в' => 'v', 'В' => 'v', 'г' => 'g', 'Г' => 'g', 'д' => 'd', 'Д' => 'd', 'е' => 'e', 'Е' => 'E', 'ё' => 'e', 'Ё' => 'E', 'ж' => 'zh', 'Ж' => 'zh', 'з' => 'z', 'З' => 'z', 'и' => 'i', 'И' => 'i', 'й' => 'j', 'Й' => 'j', 'к' => 'k', 'К' => 'k', 'л' => 'l', 'Л' => 'l', 'м' => 'm', 'М' => 'm', 'н' => 'n', 'Н' => 'n', 'о' => 'o', 'О' => 'o', 'п' => 'p', 'П' => 'p', 'р' => 'r', 'Р' => 'r', 'с' => 's', 'С' => 's', 'т' => 't', 'Т' => 't', 'у' => 'u', 'У' => 'u', 'ф' => 'f', 'Ф' => 'f', 'х' => 'h', 'Х' => 'h', 'ц' => 'c', 'Ц' => 'c', 'ч' => 'ch', 'Ч' => 'ch', 'ш' => 'sh', 'Ш' => 'sh', 'щ' => 'sch', 'Щ' => 'sch', 'ъ' => '', 'Ъ' => '', 'ы' => 'y', 'Ы' => 'y', 'ь' => '', 'Ь' => '', 'э' => 'e', 'Э' => 'e', 'ю' => 'ju', 'Ю' => 'ju', 'я' => 'ja', 'Я' => 'ja');
        
        $text = str_replace(array_keys($replacements), array_values($replacements), $text);

        // Remove any character that is not alphanumeric, white-space, or a hyphen 
        $text = preg_replace('/[^a-z0-9\s\-\_]/i', '', $text);

        return trim($text);
    }
}
