<?php

namespace AppBundle\Form\Type;

use AppBundle\Entity\Menu;
use AppBundle\Entity\Module;
use AppBundle\Enum\ModuleType as ModuleTypeEnum;
use AppBundle\Enum\TypeTemplateTitle;
use AppBundle\Helper\ModuleHelper;
use Doctrine\ORM\EntityRepository;
use Symfony\Bridge\Doctrine\Form\Type\EntityType;
use Symfony\Component\Form\AbstractType;
use Symfony\Component\Form\Extension\Core\DataTransformer\NumberToLocalizedStringTransformer;
use Symfony\Component\Form\Extension\Core\Type\CheckboxType;
use Symfony\Component\Form\Extension\Core\Type\ChoiceType;
use Symfony\Component\Form\Extension\Core\Type\FileType;
use Symfony\Component\Form\Extension\Core\Type\HiddenType;
use Symfony\Component\Form\Extension\Core\Type\NumberType;
use Symfony\Component\Form\Extension\Core\Type\TextType;
use Symfony\Component\Form\Extension\Core\Type\TextareaType;
use Symfony\Component\Form\FormBuilderInterface;
use Symfony\Component\Form\FormEvent;
use Symfony\Component\Form\FormEvents;
use Symfony\Component\OptionsResolver\OptionsResolver;

class ModuleType extends AbstractType
{
    /**
     * 
     * @param FormBuilderInterface $builder 
     * @param array $options 
     */
    public function buildForm(FormBuilderInterface $builder, array $options)
    {
        $positionList = ModuleHelper::getPositionList();
        $layoutList = $options['data']->getLayoutList();

        $builder
            ->add('title', TextType::class, array(
                'label' => 'form.module.default.title.label',
                'required' => true,
            ))
            ->add('position', ChoiceType::class, array(
                'choices' => $positionList,
                'preferred_choices' => array('sidebar_right'),
                'choice_label' => function ($choiceValue, $key, $value) {
                    return 'form.module.default.position.'.$key;
                },
                'label' => 'form.module.default.position.label',
                'required' => true,
            ))
            ->add('type', HiddenType::class, array(
                'required' => true,
            ))
            ->add('layout', ChoiceType::class, array(
                'choices' => $layoutList,
                'preferred_choices' => array('default.html.twig'),
                'choice_label' => function ($choiceValue, $key, $value) {
                    $tmp = str_replace('.html.twig', '', $key);
                    if($tmp == 'cols') {
                        $tmp = 'cols.default';
                    }
                    return 'form.module.default.layout.'.$tmp;
                },
                'label' => 'form.module.default.layout.label',
                'required' => true,
            ))
            ->add('custom_class', TextType::class, array(
                'label' => 'form.module.default.custom_class.label',
                'required' => false,
            ))
            ->add('ordering', NumberType::class, array(
                'label' => 'form.module.default.order.label',
                'required' => false,
                'rounding_mode' => NumberToLocalizedStringTransformer::ROUND_UP
            ))
            ->add('is_active', CheckboxType::class, array(
                'label' => 'form.module.default.is_active.label',
                'required' => false,
            ))
            ->add('menu', EntityType::class, array(
                'class' => Menu::class,
                'choice_label'  => function ($menu) {
                    $label = $menu->getTitle();
                    $parents = $menu->getParents();

                    if (count($parents) > 0) {
                        $tmp = $menu->getTitle();
                        foreach ($parents as $parent) {
                            $tmp = $parent->getTitle() . ' / ' . $tmp;
                        }
                        $label .= ' <small><em>( ' . $tmp . ' )</em></small>';
                    }
                    return $label;
                },
                'expanded' => true,
                'multiple' => true,
                'label' => 'form.module.default.menu.label',
                'placeholder' => 'form.module.default.menu.placeholder',
                'query_builder' => function(EntityRepository $er) {
                    //return $er->createQueryBuilder('m')->orderBy('m.title', 'ASC');
                    $qb = $er->createQueryBuilder('m');
                    return $qb
                            ->leftJoin('m.typeTemplate', 'tt')
                            ->leftJoin('m.content', 'content')
                            ->where('m.is_active = :m_is_active')
                            ->andWhere('tt.title NOT IN (:m_typetemplate)')
                            ->andWhere('content IS NOT NULL')
                            ->orderBy('m.title', 'ASC')
                            ->setParameters(array(
                                'm_is_active' => true,
                                'm_typetemplate' => array(
                                    TypeTemplateTitle::SEPARATOR
                                )
                            ));
                },
                'required'  => true,
            ))
            ->addEventListener(FormEvents::PRE_SET_DATA, array($this, 'onPreSetData'))
        ;
    }

    /**
     * Please use it only for picture column
     * Use custom Form AppBundle\Form\Type\Module\NewModuleType for custom module params
     */
    public function onPreSetData(FormEvent $event)
    {
        $form = $event->getForm();
        
        $module = $event->getData();
        $formClass = $module->getFormClass();
        
        switch ($formClass) {
            case "CustomHtml" :
                                $form
                                    ->add('filePicture', FileType::class, array(
                                        'attr' => array('class' => 'image_input'),
                                        'image_path' => 'webPicture',
                                        'label' => 'form.module.custom_html.picture.label',
                                        'help' => 'form.module.custom_html.picture.help',
                                        'required' => false,
                                    ))
                                ;
                                break;
            default :
                                break;                
        }
    }

    /**
     * 
     * @param OptionsResolver $resolver 
     */
    public function configureOptions(OptionsResolver $resolver)
    {
        $resolver->setDefaults(array(
            'data_class' => Module::class,
            'translation_domain' => 'form',
        ));
    }
}
