<?php

namespace AppBundle\Mail\Model;

use AppBundle\Entity\Document;
use Doctrine\Common\Collections\ArrayCollection;
use Doctrine\Common\Collections\Collection;

/**
 * Mail
 */
class Mail
{
    /**
     * @var array List of arrays. Each array must have a mail adress as key, and null or mail label as value
     */
    private $cc = array();

    /**
     * @var string
     */
    private $charset = 'utf-8';

    /**
     * @var string
     */
    private $contentType = 'text/html';

    /**
     * Describe the mail.
     * This information will not be part of the mail send to the addresses.
     * @var string
     */
    private $description = '';

    /**
     * @var array Key is the mail adress, value the mail label. Only one element (one mail sender)
     */
    private $from = array();

    /**
     * @var string Mail body, HTML-formatted
     */
    private $html = '';

    /**
     * @var string
     */
    private $subject = '';

    /**
     * @var string Mail body, text-formatted
     */
    private $text = '';

    /**
     * @var array List of arrays. Each array must have a mail adress as key, and null or mail label as value
     */
    private $to = array();

    /**
     * @var ArrayCollection
     */
    private $attachments;


    /**
     * 
     * @param string $subject
     */
    public function __construct($subject = '')
    {
        $this->attachments = new ArrayCollection();
        $this->subject = $subject;
    }

    public function __clone()
    {
        $this->attachments = new ArrayCollection();
    }


    /**
     * Set html
     *
     * @param string $html
     *
     * @return Mail
     */
    public function setHtml(string $html): Mail
    {
        $this->html = $html;

        return $this;
    }

    /**
     * Get html
     *
     * @return string
     */
    public function getHtml(): string
    {
        return $this->html;
    }


    /**
     * Set text
     *
     * @param string $text
     *
     * @return Mail
     */
    public function setText(string $text): Mail
    {
        $this->text = $text;

        return $this;
    }

    /**
     * Get text
     *
     * @return string
     */
    public function getText(): string
    {
        return $this->text;
    }

    /**
     * Add cc
     *
     * @param string $ccMail
     * @param string|null $ccLabel
     *
     * @return Mail
     */
    public function addCc(string $ccMail, $ccLabel = null): Mail
    {
        $this->cc[$ccMail] = $ccLabel;

        return $this;
    }

    /**
     * Get cc
     *
     * @return array
     */
    public function getCc(): array
    {
        return $this->cc;
    }

    /**
     * Set charset
     *
     * @param string $charset
     *
     * @return Mail
     */
    public function setCharset(string $charset): Mail
    {
        $this->charset = $charset;

        return $this;
    }

    /**
     * Get charset
     *
     * @return string
     */
    public function getCharset(): string
    {
        return $this->charset;
    }

    /**
     * Set contentType
     *
     * @param string $contentType
     *
     * @return Mail
     */
    public function setContentType(string $contentType): Mail
    {
        $this->contentType = $contentType;

        return $this;
    }

    /**
     * Get contentType
     *
     * @return string
     */
    public function getContentType(): string
    {
        return $this->contentType;
    }

    /**
     * Set description
     *
     * @param string $description
     *
     * @return Mail
     */
    public function setDescription(string $description): Mail
    {
        $this->description = $description;

        return $this;
    }

    /**
     * Get description
     *
     * @return string
     */
    public function getDescription(): string
    {
        return $this->description;
    }

    /**
     * Set from
     *
     * @param string $fromMail
     * @param mixed $fromLabel
     *
     * @return Mail
     */
    public function setFrom(string $fromMail, $fromLabel = null): Mail
    {
        $this->from = []; // Reset field (only one sender)

        $this->from[$fromMail] = $fromLabel;

        return $this;
    }

    /**
     * Get from
     *
     * @return array
     */
    public function getFrom(): array
    {
        return $this->from;
    }

    /**
     * Set subject
     *
     * @param string $subject
     *
     * @return Mail
     */
    public function setSubject(string $subject): Mail
    {
        $this->subject = $subject;

        return $this;
    }

    /**
     * Get subject
     *
     * @return string
     */
    public function getSubject(): string
    {
        return $this->subject;
    }

    /**
     * Add to
     *
     * @param string $toMail
     * @param string|null $toLabel
     *
     * @return Mail
     */
    public function addTo(string $toMail, $toLabel = null): Mail
    {
        $this->to[$toMail] = $toLabel;

        return $this;
    }

    /**
     * Get to
     *
     * @return array
     */
    public function getTo(): array
    {
        return $this->to;
    }

    /**
     * Add attachment
     * @param Document $document
     * @return Mail
     */
    public function addAttachment(Document $document): Mail
    {
        $this->attachments[] = $document;

        return $this;
    }

    /**
     * Remove attachment
     * @param  Document $document 
     */
    public function removeAttachment(Document $document): void
    {
        $this->attachments->removeElement($document);
    }

    /**
     * Get attachments
     * @return Collection
     */
    public function getAttachments(): Collection
    {
        return $this->attachments;
    }
}
