<?php

namespace AppBundle\Mail;

use AppBundle\Entity\Availability;
use AppBundle\Entity\OperationITDLC;
use AppBundle\Entity\Media;
use AppBundle\Mail\Model\Mail;
use AppBundle\Enum\OperationITDLCType;
use AppBundle\Enum\UserType;
use AppBundle\Enum\MediaType;
use AppBundle\Entity\User;
use AppBundle\Entity\MerITDLC;
use Symfony\Component\Routing\Generator\UrlGeneratorInterface;
use Psr\Log\LogLevel;

class ItdlcMailer extends AbstractMailer
{
  
    
    /**
     * Notify the user that his application for an operation is confirmed.
     *
     * @param User $user
     *
     * @return array Mailer response
     * @throws \Exception
     */
    public function confirmDemande(User $user): array
    {
        $responses = [];

        switch ($user->getType()) {
            case UserType::ENGINEER:
                $mailEngineer = new Mail('Confirmation de demande de participation à l’action Ingénieurs et techniciens dans les classes');
                $mailEngineer->setFrom($this->getParameter('itdlc'), $this->getParameter('label'));
                $mailEngineer->addTo($user->getEmail(), $user->getFullname());

                $mailEngineer->setHtml(
                    $this->renderView('Admin/Mail/ITDLC/addDemandeEngineer.html.twig', [
                        'firstname' => $user->getFullName()
                    ])
                );

                $responses = $this->send($mailEngineer);
                break;

            case UserType::TEACHER:
                $mailTeacher = new Mail('[CGénial] Trouvez l’intervenant·e qu’il vous faut !');
                $mailTeacher->setFrom($this->getParameter('itdlc'), $this->getParameter('label'));
                $mailTeacher->addTo($user->getEmail(), $user->getFullname());
                if ($user->getBossemail()) {
                    $mailTeacher->addCc($user->getBossemail());
                }

                $mailTeacher->setHtml(
                    $this->renderView('Admin/Mail/ITDLC/addDemandeProf.html.twig', [
                        'firstname' => $user->getFullName(),
                        'mailto' => $this->getParameter('itdlc'),
                    ])
                );

                $responses = $this->send($mailTeacher);
                break;

            default:
                break;
        }

        return $responses;
    }

    /**
     * Suggest an engineer the availabilities of a teacher for a given ITDLC operation
     * @param OperationITDLC $demandeItdlc
     * @param User $engineer
     * @return Mail
     * @throws \Exception
     */
    public function getSuggestOperationMeeting(OperationITDLC $demandeItdlc, User $engineer): Mail
    {
        // Get the teacher's availabilities
        $merITDLC = $this->em->createQueryBuilder();
        $merITDLC
            ->select('m')
            ->from(MerITDLC::class, 'm')
            ->where('m.ope = :ope_id')
            ->setParameter('ope_id', $demandeItdlc->getId());
           
        $mer = $merITDLC->getQuery()->getOneOrNullResult();

        // Create engineer's mail
        $mailEngineer = new Mail('[CGénial] Proposition de rencontre en classe');
        $mailEngineer->setDescription($this->translator->trans('word.type.user.engineer.label', [], 'word'));
        $mailEngineer->setFrom($this->getParameter('itdlc'), $this->getParameter('label'));
        $mailEngineer->addTo($engineer->getEmail(), $engineer->getFullname());
        $mailEngineer->addCc($this->getParameter('itdlc'), $this->getParameter('label'));
        $mailEngineer->setHtml(
            $this->renderView('Admin/Mail/ITDLC/suggestOperationMeeting.html.twig', [
                'engineer' => $engineer,
                'demandeItdlc' => $demandeItdlc,
                'teacher' => $demandeItdlc->getProf(),
                'mer' => $mer,
            ])
        );

        return $mailEngineer;
    }

    /**
     * Get the emails related to the state of an operation.
     * As we cannot use only 1 method to handle every cases (the code will be hundreds of lines long),
     * This method calls the dedicated method, according to the type of the operation (operation or application) and the new state.
     * @param OperationITDLC $operationItdlc
     * @param string $statePrevious
     * @return array Mail[]
     * @throws \Exception
     */
    public function getOperationState(OperationITDLC $operationItdlc, string $statePrevious): array
    {
        $methodName = 'get'
            . (OperationITDLCType::OPERATION === $operationItdlc->getType() ? 'Operation' : 'Application')
            . 'State'
            . ucfirst($operationItdlc->getState());

        if (!method_exists($this, $methodName)) {
            throw new \Exception('Cannot get emails related to the given ITDLC operation: there is no method called ' . $methodName);
        }

        return $this->$methodName($operationItdlc, $statePrevious);
    }

    /**
     * Send a mail to the teacher and the engineer after the closing of an ITDLC operation
     *
     * @param OperationITDLC $operationItdlc
     * @param string $statePrevious
     * @return array Mail[]
     * @throws \Exception
     */
    private function getOperationStateClosed(OperationITDLC $operationItdlc, string $statePrevious): array
    {
        $engineer = $operationItdlc->getInge();
        $teacher = $operationItdlc->getProf();

        // Get the availabily for this operation
        $availability = $this->em->getRepository(Availability::class)->findOneBy(['operationitdlc' => $operationItdlc]);

        // Create mails
        $mailTeacher = new Mail(
            sprintf('[%s] Suite à votre rencontre Ingénieurs et techniciens dans les classes', $this->getParameter('label'))
        );
        $mailTeacher->setFrom($this->getParameter('itdlc'), $this->getParameter('label'));

        $mailEngineer = clone $mailTeacher;

        // Customize teacher's mail
        $mailTeacher->setDescription($this->translator->trans('word.type.user.teacher.label', [], 'word'));
        $mailTeacher->addTo($teacher->getEmail(), $teacher->getFullname());
        $mailTeacher->setHtml(
            $this->renderView('Admin/Mail/ITDLC/endOperation.html.twig', [
                'firstname' => $teacher->getFirstname(),
                'operationItdlc' => $operationItdlc,
                'availability' => $availability,
                'lienquestionnaire' => 'https://docs.google.com/forms/d/1UB5cemz4DSpCfFey7kjvx1EelNsR7hFT_ozEo9Zofbo/viewform',
            ])
        );

        // Customize engineer's mail
        $mailEngineer->setDescription($this->translator->trans('word.type.user.engineer.label', [], 'word'));
        $mailEngineer->addTo($engineer->getEmail(), $engineer->getFullname());
        $mailEngineer->setHtml(
            $this->renderView('Admin/Mail/ITDLC/endOperation.html.twig', [
                'firstname' => $engineer->getFirstname(),
                'operationItdlc' => $operationItdlc,
                'availability' => $availability,
                'lienquestionnaire' => 'https://docs.google.com/forms/d/1U3Y1k1odwQghWA9NUcx49Cf2IZ2dxkWn8c48d_80RlQ/viewform',
            ])
        );

        return [$mailTeacher, $mailEngineer];
    }

    /**
     * Send a mail to the teacher if there is no contributor for this application.
     *
     * @param OperationITDLC $operationItdlc
     * @param string $statePrevious
     * @return array Mail[]
     * @throws \Exception
     */
    private function getApplicationStateNoContributor(OperationITDLC $operationItdlc, string $statePrevious): array
    {
        $teacher = $operationItdlc->getProf();

        // Get the availabily for this operation
        $availability = $this->em->getRepository(Availability::class)->findOneBy(['operationitdlc' => $operationItdlc]);

        // Create mails
        $mailTeacher = new Mail(
            sprintf('[%s] Suite à votre rencontre Ingénieurs et techniciens dans les classes', $this->getParameter('label'))
        );
        $mailTeacher->setFrom($this->getParameter('itdlc'), $this->getParameter('label'));

        // Customize teacher's mail
        $mailTeacher->setDescription($this->translator->trans('word.type.user.teacher.label', [], 'word'));
        $mailTeacher->addTo($teacher->getEmail(), $teacher->getFullname());
        $mailTeacher->setHtml(
            $this->renderView('Admin/Mail/ITDLC/noContributorApplication.html.twig', [
                'firstname' => $teacher->getFirstname(),
                'operationItdlc' => $operationItdlc,
                'availability' => $availability,
                'lienquestionnaire' => 'https://docs.google.com/forms/d/1UB5cemz4DSpCfFey7kjvx1EelNsR7hFT_ozEo9Zofbo/viewform',
            ])
        );

        return [$mailTeacher];
    }

    /**
     * Notify the teacher and the engineer of an ITDLC operation that the operation is now open
     *
     * @param OperationITDLC $operationItdlc
     * @return array List of Mail instances
     * @throws \Exception
     */
    public function getConfirmOperation(OperationITDLC $operationItdlc): array
    {
        $engineer = $operationItdlc->getInge();
        $teacher = $operationItdlc->getProf();

        // Get the availabily for this operation
        $availability = $this->em->getRepository(Availability::class)->findOneBy(['operationitdlc' => $operationItdlc]);
        
        $videos = $this->em->getRepository(Media::class)->findBy([
            'is_active' => true,
            'opeItdlcEmail' => UserType::ENGINEER,
            'type' => MediaType::VIDEO], [
            'title' => 'ASC', 'id' => 'ASC']
        );

        $files = $this->em->getRepository(Media::class)->findBy([
            'is_active' => true,
            'opeItdlcEmail' => UserType::ENGINEER,
            'type' => MediaType::PDF ], [
            'title' => 'ASC', 'id' => 'ASC']
        );


        // Create mail
        $mailEngineer = new Mail(
            sprintf('[%s] Confirmation de rencontre Ingénieurs et techniciens dans les classes', $this->getParameter('label'))
        );
        $mailEngineer->setFrom($this->getParameter('itdlc'), $this->getParameter('label'));
        $mailTeacher = clone $mailEngineer;

        // Customize engineer's mail
        $mailEngineer->setDescription($this->translator->trans('word.type.user.engineer.label', [], 'word'));
        $mailEngineer->addTo($engineer->getEmail(), $engineer->getFullname());
        $mailEngineer->setHtml(
            $this->renderView('Admin/Mail/ITDLC/newOperation.html.twig', [
                'addressee' => $engineer,
                'availability' => $availability,
                'files' => $files ?? [],
                'videos' => $videos ?? [],
                'emailRegistrationItdlc' => $this->getParameter('itdlc'),
                'engineer' => $engineer,
                'operationItdlc' => $operationItdlc,
                'teacher' => $teacher,
            ])
        );


        $videos = $this->em->getRepository(Media::class)->findBy([
            'is_active' => true,
            'opeItdlcEmail' => UserType::TEACHER,
            'type' => MediaType::VIDEO], [
            'title' => 'ASC', 'id' => 'ASC']
        );

        $files = $this->em->getRepository(Media::class)->findBy([
            'is_active' => true,
            'opeItdlcEmail' => UserType::TEACHER,
            'type' => MediaType::PDF ], [
            'title' => 'ASC', 'id' => 'ASC']
        );

        // Customize teacher's mail
        $mailTeacher->setDescription($this->translator->trans('word.type.user.teacher.label', [], 'word'));
        $mailTeacher->addTo($teacher->getEmail(), $teacher->getFullname());
        if ($teacher->getBossemail()) {
            $mailTeacher->addCc($teacher->getBossemail());
        }
        $mailTeacher->setHtml(
            $this->renderView('Admin/Mail/ITDLC/newOperation.html.twig', [
                'availability' => $availability,
                'addressee' => $teacher,
                'files' => $files ?? [],
                'videos' => $videos ?? [],
                'emailRegistrationItdlc' => $this->getParameter('itdlc'),
                'engineer' => $engineer,
                'teacher' => $teacher,
                'operationItdlc' => $operationItdlc,
            ])
        );

        return [$mailEngineer, $mailTeacher];
    }


    /**
     * Notify Admin - Operation cancelled
     *
     * @param OperationITDLC $ope
     * @param User $user
     *
     * @return array Mailer response
     * @throws \Exception
     */
    public function cancelOperation(OperationITDLC $ope, User $user, $type): array
    {
        $responses = [];
        
        $urlEspacePerso = $this->getParameter('domain') . $this->router->generate('espace_perso');


        $mailAdmin = new Mail('Annulation d\'un participant à une action Ingénieurs et techniciens dans les classes');
        $mailAdmin->setFrom($this->getParameter('itdlc_registration'), $this->getParameter('label'));
        $mailAdmin->addTo($this->getParameter('itdlc_registration'), $this->getParameter('label'));

        $mailAdmin->setHtml(
            $this->renderView('Front/Mail/ITDLC/cancelOperationAdmin.html.twig', [
                'user' => $user,
                'ope' => $ope,
            ])
        );

        $responses[] = $this->send($mailAdmin);

        if ($type == UserType::TEACHER) {
            $mailto = $ope->getInge()->getEmail();
            $label = $ope->getInge()->getFullname();
    
            $mail = new Mail("Annulation de l'intervention en classe programmée");
            $mail->setFrom($this->getParameter('itdlc_registration'), $this->getParameter('label'));
            $mail->addTo($mailto, $label);
    
            $mail->setHtml(
                $this->renderView('Front/Mail/ITDLC/cancelOperationEngineer.html.twig', [
                    'user' => $user,
                    'ope' => $ope,
                ])
            );
        } elseif ($type == UserType::ENGINEER) {
            $mailto = $ope->getProf()->getEmail();
            $label = $ope->getProf()->getFullname();
    
            $mail = new Mail("Annulation de l'intervention en classe programmée par l'intervenant");
            $mail->setFrom($this->getParameter('itdlc_registration'), $this->getParameter('label'));
            $mail->addTo($mailto, $label);
    
            $mail->setHtml(
                $this->renderView('Front/Mail/ITDLC/cancelOperationTeacher.html.twig', [
                    'user' => $user,
                    'ope' => $ope,
                    'urlEspacePerso' => $urlEspacePerso
                ])
            );
        }
       

        $responses[] = $this->send($mail);

        return $responses;
    }


     /**
      * Notify the engineer that the teacher has chosen him or her for an ITDLC operation.
      *
      * @param MerITDLC $mer
      * @return array List of Mail instances
      * @throws \Exception
      */
    public function sendMerToInge(MerITDLC $mer): array
    {
        $engineer = $mer->getInge();
        $ope = $mer->getOpe();

        $urlEspacePerso = $this->getParameter('domain') . $this->router->generate('espace_perso');

        // Get the availabily for this operation
        $dates = $mer->getAllDates();

        // Create mail
        $mailEngineer = new Mail(
            sprintf('[%s] %s est intéressé·e par votre profil', $this->getParameter('label'), $ope->getProf()->getFullName())
        );

        $mailEngineer->setFrom($this->getParameter('itdlc'), $this->getParameter('label'));

        // Customize engineer's mail
        $mailEngineer->setDescription($this->translator->trans('word.type.user.engineer.label', [], 'word'));
        $mailEngineer->addTo($engineer->getEmail(), $engineer->getFullname());
        $mailEngineer->setHtml(
            $this->renderView('Front/Mail/ITDLC/merITDLC.html.twig', [
                'dates' => $dates,
                'engineer' => $engineer,
                'ope' => $ope,
                'urlEspacePerso' => $urlEspacePerso
            ])
        );

        $response = $this->send($mailEngineer);

        return $response;
    }

    /**
      * Notify the teacher about his or her ITDLC operation.
      *
      * @param OperationITDLC $mer
      * @return array List of Mail instances
      * @throws \Exception
      */
    public function sendOperationNoMerToTeacher(OperationITDLC $ope): array
    {
        $teacher = $ope->getProf();

        $urlEspacePerso = $this->getParameter('domain') . $this->router->generate('espace_perso');

        // Create mail
        $mailTeacher = new Mail(
            sprintf('[%s] Suite à votre demande d’intervention', $this->getParameter('label'))
        );

        $mailTeacher->setFrom($this->getParameter('itdlc'), $this->getParameter('label'));

        // Customize engineer's mail
        $mailTeacher->setDescription($this->translator->trans('word.type.user.engineer.label', [], 'word'));
        $mailTeacher->addTo($teacher->getEmail(), $teacher->getFullname());
        $mailTeacher->setHtml(
            $this->renderView('Front/Mail/ITDLC/noMerITDLC.html.twig', [
                'ope' => $ope,
                'urlEspacePerso' => $urlEspacePerso
            ])
        );

        return $this->send($mailTeacher);

    }

    /**
     * Notify the teacher and the engineer that the ITDLC is accepted
     *
     * @param MerITDLC $mer
     * @return array List of Mail responses
     * @throws \Exception
     */
    public function sendITDLCAccepted(MerITDLC $mer): array
    {
        $responses = array();
        $engineer = $mer->getInge();
        $ope = $mer->getOpe();
        $teacher = $ope->getProf();

        // Create mail Teacher
        $mailTeacher = new Mail(
            sprintf('[%s] %s a accepté votre proposition', $this->getParameter('label'), $engineer->getFullName())
        );

        $mailTeacher->setFrom($this->getParameter('itdlc'), $this->getParameter('label'));

        $videos = $this->em->getRepository(Media::class)->findBy([
            'is_active' => true,
            'opeItdlcEmail' => UserType::TEACHER,
            'type' => MediaType::VIDEO], [
            'title' => 'ASC', 'id' => 'ASC']
        );

        $files = $this->em->getRepository(Media::class)->findBy([
            'is_active' => true,
            'opeItdlcEmail' => UserType::TEACHER,
            'type' => MediaType::PDF ], [
            'title' => 'ASC', 'id' => 'ASC']
        );


        // Customize engineer's mail
        $mailTeacher->setDescription($this->translator->trans('word.type.user.engineer.label', [], 'word'));
        $mailTeacher->addTo($teacher->getEmail(), $teacher->getFullname());
        $mailTeacher->setHtml(
            $this->renderView('Front/Mail/ITDLC/acceptedMerITDLC.html.twig', [
                'files' => $files ?? [],
                'videos' => $videos ?? [],
                'availability' => $ope->getAvailability()->last(),
                'ope' => $ope,
                'type' => "teacher"
            ])
        );

        $responses["teacher"] = $this->send($mailTeacher);

        // Create mail Engineer
        $mailEngineer = new Mail(
            sprintf('[%s] Confirmation Rencontre en classe', $this->getParameter('label'))
        );

        $mailEngineer->setFrom($this->getParameter('itdlc'), $this->getParameter('label'));
        $videos = $this->em->getRepository(Media::class)->findBy([
            'is_active' => true,
            'opeItdlcEmail' => UserType::ENGINEER,
            'type' => MediaType::VIDEO], [
            'title' => 'ASC', 'id' => 'ASC']
        );

        $files = $this->em->getRepository(Media::class)->findBy([
            'is_active' => true,
            'opeItdlcEmail' => UserType::ENGINEER,
            'type' => MediaType::PDF ], [
            'title' => 'ASC', 'id' => 'ASC']
        );


        // Customize engineer's mail
        $mailEngineer->setDescription($this->translator->trans('word.type.user.engineer.label', [], 'word'));
        $mailEngineer->addTo($engineer->getEmail(), $engineer->getFullname());
        $mailEngineer->setHtml(
            $this->renderView('Front/Mail/ITDLC/acceptedMerITDLC.html.twig', [
                'files' => $files ?? [],
                'videos' => $videos ?? [],
                'availability' => $ope->getAvailability()->last(),
                'ope' => $ope,
                'type' => "engineer"
            ])
        );

        $responses["engineer"] = $this->send($mailEngineer);

        return $responses;
    }

    /**
     * Notify the teacher and the engineer that the ITDLC is coming soon
     *
     * @param OperationITDLC $ope
     * @return array List of Mail responses
     * @throws \Exception
     */
    public function sendReminderComingSoonOperation(OperationITDLC $ope): array
    {
        $responses = array();
        $engineer = $ope->getInge();
        if($engineer !== null)
        {
            $teacher = $ope->getProf();

            $urlEspacePerso = $this->getParameter('domain') . $this->router->generate('espace_perso');

            // Create mail Teacher
            $mailTeacher = new Mail(
                sprintf('[%s] Rappel de l’intervention', $this->getParameter('label'), $engineer->getFullName())
            );

            $mailTeacher->setFrom($this->getParameter('itdlc'), $this->getParameter('label'));

            // Customize engineer's mail
            $mailTeacher->setDescription($this->translator->trans('word.type.user.engineer.label', [], 'word'));
            $mailTeacher->addTo($teacher->getEmail(), $teacher->getFullname());

            $mailTeacher->setHtml(
                $this->renderView('Front/Mail/ITDLC/remindersOperationITDLC.html.twig', [
                    'availability' => $ope->getAvailability()->last(),
                    'ope' => $ope,
                    'type' => "teacher",
                    'urlEspacePerso' => $urlEspacePerso
                ])
            );

            $responsesMailTeacher = $this->send($mailTeacher);

            // Create mail Engineer
            $mailEngineer = new Mail(
                sprintf('[%s] Rappel de l’intervention', $this->getParameter('label'))
            );

            $mailEngineer->setFrom($this->getParameter('itdlc'), $this->getParameter('label'));

            // Customize engineer's mail
            $mailEngineer->setDescription($this->translator->trans('word.type.user.engineer.label', [], 'word'));
            $mailEngineer->addTo($engineer->getEmail(), $engineer->getFullname());
            $mailEngineer->setHtml(
                $this->renderView('Front/Mail/ITDLC/remindersOperationITDLC.html.twig', [
                    'availability' => $ope->getAvailability()->last(),
                    'ope' => $ope,
                    'type' => "engineer",
                    'urlEspacePerso' => $urlEspacePerso
                ])
            );

            $responsesMailEngineer = $this->send($mailEngineer);
        }
        else
        {
            $responsesMailEngineer = null;
            $responsesMailTeacher = null;
        }

        return [$responsesMailTeacher, $responsesMailEngineer];
    }


     /**
      * Notify the engineer that the teacher is waiting an answer about the MER since 5 days.
      *
      * @param MerITDLC $mer
      * @return array List of Mail instances
      * @throws \Exception
      */
      public function sendReminderMerToInge(MerITDLC $mer): array
      {
          $engineer = $mer->getInge();
          if($engineer !== null)
          {
            $ope = $mer->getOpe();

            $urlEspacePerso = $this->getParameter('domain') . $this->router->generate('espace_perso');

            // Get the availabily for this operation
            $dates = $mer->getAllDates();

            // Create mail
            $mailEngineer = new Mail(
                sprintf('RAPPEL - [%s] %s est intéressé·e par votre profil', $this->getParameter('label'), $ope->getProf()->getFullName())
            );

            $mailEngineer->setFrom($this->getParameter('itdlc'), $this->getParameter('label'));

            // Customize engineer's mail
            $mailEngineer->setDescription($this->translator->trans('word.type.user.engineer.label', [], 'word'));
            $mailEngineer->addTo($engineer->getEmail(), $engineer->getFullname());
            $mailEngineer->setHtml(
                $this->renderView('Front/Mail/ITDLC/remindersMerITDLC.html.twig', [
                    'dates' => $dates,
                    'engineer' => $engineer,
                    'ope' => $ope,
                    'urlEspacePerso' => $urlEspacePerso
                ])
            );
          } else {
              $response = [];
          }

          $response = $this->send($mailEngineer);

          return $response;
      }


    /**
     * Notify the teacher that the engineer don't answer about the MER since 10 days.
     *
     * @param MerITDLC $mer
     * @return array List of Mail instances
     * @throws \Exception
     */
      public function sendReminderMerToTeacher(MerITDLC $mer): array
      {
        $ope = $mer->getOpe();
        $teacher = $ope->getProf();

        if($ope->getInge() !== null)
        {
            

            $urlEspacePerso = $this->getParameter('domain') . $this->router->generate('espace_perso');

            // Create mail
            $mailTeacher = new Mail(
                sprintf('[%s] %s n’a pas donné réponse à votre proposition', $this->getParameter('label'), $ope->getInge()->getFullName())
            );

            // Get the 3 closest engineers
            $engineers = $this->em->getRepository(User::class)->findClosestOthers($teacher, array(), false, 3);

            $mailTeacher->setFrom($this->getParameter('itdlc'), $this->getParameter('label'));

            $mailTeacher->setDescription($this->translator->trans('word.type.user.engineer.label', [], 'word'));
            $mailTeacher->addTo($teacher->getEmail(), $teacher->getFullname());

            $mailTeacher->setHtml(
                $this->renderView('Front/Mail/ITDLC/noAnswerMerITDLC.html.twig', [
                    'teacher' => $teacher,
                    'ope' => $ope,
                    'engineers' => $engineers,
                    'urlEspacePerso' => $urlEspacePerso
                ])
            );

            $response = $this->send($mailTeacher);
        }
        else
        {
            $response = [];
        }
        return $response;
      }

    /**
     * Send the satifaction survey to the users
     *
     * @param string $userType
     * @param OperationITDLC $ope
     * @return array List of Mail instances with code success or failure
     * @throws \Exception
     */
    public function sendFormSatisfaction($userType, OperationITDLC $ope): array
    {

        if ($userType == "engineer") {
            $engineer = $ope->getInge();

            // Create mail
            $mailEngineer = new Mail(
                sprintf('[%s] Suite à votre rencontre en classe', $this->getParameter('label'), $ope->getProf()->getFullName())
            );

            $mailEngineer->setFrom($this->getParameter('itdlc'), $this->getParameter('label'));

            $mailEngineer->setDescription($this->translator->trans('word.type.user.engineer.label', [], 'word'));
            $mailEngineer->addTo($engineer->getEmail(), $engineer->getFullname());

            $mailEngineer->setHtml(
                $this->renderView('Front/Mail/ITDLC/satisfactionOperation.html.twig', [
                    'ope' => $ope,
                    'availability' => $ope->getAvailability()->last(),
                    'type' => 'engineer',
                    'linkForm' => "https://docs.google.com/forms/d/e/1FAIpQLSd77y_4de6q8_jz5ys1c_3w1xYx60oWXb9XsbDkLWIZAxEI8g/viewform"
                ])
            );

            return $this->send($mailEngineer);
        } else {

            $teacher = $ope->getProf();

            // Create mail
            $mailTeacher = new Mail(
                sprintf('[%s] Suite à votre rencontre Ingénieur·e·s et technicien·ne·s dans les classes', $this->getParameter('label'), $ope->getProf()->getFullName())
            );

            $mailTeacher->setFrom($this->getParameter('itdlc'), $this->getParameter('label'));

            $mailTeacher->setDescription($this->translator->trans('word.type.user.engineer.label', [], 'word'));
            $mailTeacher->addTo($teacher->getEmail(), $teacher->getFullname());

            $mailTeacher->setHtml(
                $this->renderView('Front/Mail/ITDLC/satisfactionOperation.html.twig', [
                    'ope' => $ope,
                    'availability' => $ope->getAvailability()->last(),
                    'type' => 'teacher',
                    'linkForm' => "https://docs.google.com/forms/d/e/1FAIpQLSdxLjxbqGjZwvO3E6H0za6k0baFyKt5EQC4beplleolyZSLUA/viewform"
                ])
            );
            return $this->send($mailTeacher);
        }
    }

     /**
      * Notify the teacher that the engenier refused
      *
      * @param MerITDLC $mer
      * @return array List of Mail responses
      * @throws \Exception
      */
    public function sendITDLCRefused(MerITDLC $mer): array
    {
        $responses = array();
        $ope = $mer->getOpe();
        $teacher = $ope->getProf();

        // Create mail Teacher
        $mailTeacher = new Mail(
            sprintf('[%s] %s a décliné votre demande', $this->getParameter('label'), $ope->getInge()->getFullName())
        );

        $mailTeacher->setFrom($this->getParameter('itdlc'), $this->getParameter('label'));

        $urlEspacePerso = $this->getParameter('domain') . $this->router->generate('espace_perso');
        $mailTeacher->setDescription($this->translator->trans('word.type.user.engineer.label', [], 'word'));
        $mailTeacher->addTo($teacher->getEmail(), $teacher->getFullname());
        $mailTeacher->setHtml(
            $this->renderView('Front/Mail/ITDLC/refusedMerITDLC.html.twig', [
                'ope' => $ope,
                'mer' => $mer,
                'urlEspacePerso' => $urlEspacePerso
            ])
        );

        return $this->send($mailTeacher);
    }
}
