<?php

namespace AppBundle\Entity;

use Doctrine\Common\Collections\ArrayCollection;
use Doctrine\Common\Collections\Collection;
use Doctrine\ORM\Mapping as ORM;
use Symfony\Component\Validator\Context\ExecutionContextInterface;
use Symfony\Component\HttpFoundation\File\UploadedFile;

/**
 * SettingsOperationITDLC
 * @author Bastien Gatellier <contact@bgatellier.fr>
 *
 * @ORM\Entity(repositoryClass="AppBundle\Repository\SettingsOperationITDLCRepository")
 */
class SettingsOperationITDLC extends Settings
{
    /**
     * @var ArrayCollection
     * @ORM\ManyToMany(targetEntity="Document", cascade={"persist"}, orphanRemoval=true)
     * @ORM\JoinTable(name="settings_operationitdlc_teacher_documents",
     *      joinColumns={@ORM\JoinColumn(name="settings_operationitdlc_id", referencedColumnName="id")},
     *      inverseJoinColumns={@ORM\JoinColumn(name="document_id", referencedColumnName="id", unique=true)}
     * )
     */
    private $teacherDocuments;

    /**
     * @var ArrayCollection
     * @ORM\ManyToMany(targetEntity="Document", cascade={"persist"}, orphanRemoval=true)
     * @ORM\JoinTable(name="settings_operationitdlc_engineer_documents",
     *      joinColumns={@ORM\JoinColumn(name="settings_operationitdlc_id", referencedColumnName="id")},
     *      inverseJoinColumns={@ORM\JoinColumn(name="document_id", referencedColumnName="id", unique=true)}
     * )
     */
    private $engineerDocuments;


    /**
     * Constructor
     */
    public function __construct()
    {
        $this->teacherDocuments = new ArrayCollection();
        $this->engineerDocuments = new ArrayCollection();
    }

    /**
     * Add teacherDocuments
     *
     * @param Document $document
     * @return SettingsOperationITDLC
     */
    public function addTeacherDocument(Document $document): SettingsOperationITDLC
    {
        $this->teacherDocuments[] = $document;

        return $this;
    }

    /**
     * Remove teacherDocuments
     *
     * @param Document $document
     */
    public function removeTeacherDocument(Document $document): void
    {
        $this->teacherDocuments->removeElement($document);
    }

    /**
     * Get teacherDocuments
     *
     * @return Collection
     */
    public function getTeacherDocuments(): Collection
    {
        return $this->teacherDocuments;
    }

    /**
     * Add engineerDocuments
     *
     * @param Document $document
     * @return SettingsOperationITDLC
     */
    public function addEngineerDocument(Document $document): SettingsOperationITDLC
    {
        $this->engineerDocuments[] = $document;

        return $this;
    }

    /**
     * Remove engineerDocuments
     *
     * @param Document $document
     */
    public function removeEngineerDocument(Document $document): void
    {
        $this->engineerDocuments->removeElement($document);
    }

    /**
     * Get engineerDocuments
     *
     * @return Collection
     */
    public function getEngineerDocuments(): Collection
    {
        return $this->engineerDocuments;
    }

    /**
     * Validates that every document has a valid format:
     * 1. checks the mime type
     * 2. if the mime type is unknown (application/octet-stream), then checks the extension
     * @param  ExecutionContextInterface $context 
     * @param  [type]                    $payload                             
     */
    public function validateDocumentsFormat(ExecutionContextInterface $context, $payload)
    {
        $authorizedMimeTypes = [
            'application/pdf',
            'application/vnd.ms-powerpoint',
            'application/vnd.openxmlformats-officedocument.presentationml.presentation',
        ];
        $authorizedExtensions = [
            'pptx','ppt', 'pdf'
        ];

        foreach (['teacherDocuments', 'engineerDocuments'] as $propertyName) {
            // Iterate over the documents of the given property
            foreach ($this->{$propertyName} as $key => $document) {
                $file = $document->getFile();

                // Build a violation if the document does not have a correct mime type
                if ($file instanceof UploadedFile) {
                    $mimeType = $file->getClientMimeType();
                    $extension = $file->getClientOriginalExtension();

                    if (!in_array($mimeType, $authorizedMimeTypes) && ($mimeType !== 'application/octet-stream' || !in_array($extension, $authorizedExtensions))) {
                        $context->buildViolation('validators.settings.operationItdlc.document.mimeTypes')
                            ->atPath($propertyName . '['. $key .'].file')
                            ->addViolation();
                    }
                }
            }
        }
    }
}
