<?php

namespace AppBundle\Controller\Admin;

use AppBundle\Entity\Partner;
use AppBundle\Form\Type\PartnerType;
use Doctrine\ORM\EntityManagerInterface;
use Sensio\Bundle\FrameworkExtraBundle\Configuration\Method;
use Sensio\Bundle\FrameworkExtraBundle\Configuration\Route;
use Symfony\Bundle\FrameworkBundle\Controller\Controller;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\Translation\TranslatorInterface;

/**
 * @Route("/settings")
 */
class PartnerController extends Controller
{
    /**
     * @var EntityManagerInterface
     */
    private $em;

    /**
     * @var TranslatorInterface
     */
    private $translator;
    
    public function __construct(EntityManagerInterface $em, TranslatorInterface $translator)
    {
        $this->em = $em;
        $this->translator = $translator;
    }

    /**
     * Lists all partner entities.
     *
     * @Route(
     *     "/partner/{page}",
     *     requirements={"page": "\d+"},
     *     name="partner_index"
     * )
     *
     * @Method("GET")
     *
     * @param Request $request
     * @param int $page
     * @return \Symfony\Component\HttpFoundation\Response
     * @throws \Exception
     */
    public function indexAction(Request $request, int $page = 1)
    {
        $resultsMaxPerPage = $this->getParameter('results.admin.max_per_page');
        $criteria = [];
        if (null !== $name = $request->query->get('name', null)) {
            $criteria['name'] = ['like', $name];
        }

        $partners = $this->em->getRepository(Partner::class)->paginate(
            $criteria,
            ['name' => 'ASC'],
            $resultsMaxPerPage,
            ($page - 1) * $resultsMaxPerPage
        );

        return $this->render('Admin/Settings/partner/index.html.twig', [
            'partners' => $partners,
            'nb_pages' => ceil(count($partners) / $resultsMaxPerPage),
        ]);
    }

    /**
     * Creates a new partner entity.
     *
     * @Route("/partner/add", name="partner_add")
     * @Method({"GET", "POST"})
     *
     * @param Request $request
     * @return \Symfony\Component\HttpFoundation\RedirectResponse|\Symfony\Component\HttpFoundation\Response
     */
    public function addAction(Request $request)
    {
        $partner = new Partner();
        $form = $this->createForm(PartnerType::class, $partner);
        $form->handleRequest($request);

        if ($form->isSubmitted() && $form->isValid()) {
            $this->em->persist($partner);
            $this->em->flush();

            return $this->redirectToRoute('partner_edit', ['id' => $partner->getId()]);
        }

        return $this->render('Admin/Settings/partner/add.html.twig', [
            'partner' => $partner,
            'form' => $form->createView(),
        ]);
    }

    /**
     * Displays a form to edit an existing partner entity.
     *
     * @Route("/partner/edit/{id}", name="partner_edit")
     * @Method({"GET", "POST"})
     *
     * @param Request $request
     * @param Partner $partner The partner entity
     * @return \Symfony\Component\HttpFoundation\RedirectResponse|\Symfony\Component\HttpFoundation\Response
     */
    public function editAction(Request $request, Partner $partner)
    {
        $editForm = $this->createForm(PartnerType::class, $partner);
        $editForm->handleRequest($request);

        if ($editForm->isSubmitted() && $editForm->isValid()) {
            $this->em->flush();

            $this->addFlash('success', $this->translator->trans('notice.partner.updated', ['%name%' => $partner->getName()], 'notice'));

            return $this->redirectToRoute('partner_edit', ['id' => $partner->getId()]);
        }

        return $this->render('Admin/Settings/partner/edit.html.twig', [
            'partner' => $partner,
            'edit_form' => $editForm->createView(),
        ]);
    }

    /**
     * Deletes a partner entity.
     *
     * @Route("/partner/delete/{id}", name="partner_delete")
     * @Method({"GET"})
     *
     * @param Partner $partner The partner entity
     * @return \Symfony\Component\HttpFoundation\RedirectResponse
     */
    public function deleteAction(Partner $partner)
    {
        if (!$partner->isDeletable()) {
            throw $this->createAccessDeniedException($this->translator->trans('notice.partner.removed.failure', [], 'notice'));
        }

        $this->em->remove($partner);
        $this->em->flush();

        return $this->redirectToRoute('partner_index');
    }
}
