<?php

namespace AppBundle\Menu\Admin;

use AppBundle\Enum\OperationITDLCType;
use Doctrine\ORM\EntityManagerInterface;
use Knp\Menu\FactoryInterface;
use Knp\Menu\ItemInterface;
use Symfony\Component\HttpFoundation\RequestStack;
use Symfony\Component\Translation\TranslatorInterface;
use AppBundle\Enum\UserType;

class ActionItdlcBuilder
{
    /**
     * @var EntityManagerInterface
     */
    private $em;

    /**
     * @var FactoryInterface
     */
    private $factory;

    /**
     * @var null|\Symfony\Component\HttpFoundation\Request
     */
    private $request;

    /**
     * @var TranslatorInterface
     */
    private $translator;

    public function __construct(
        EntityManagerInterface $em,
        FactoryInterface $factory,
        RequestStack $requestStack,
        TranslatorInterface $translator
    )
    {
        $this->em = $em;
        $this->factory = $factory;
        $this->request = $requestStack->getCurrentRequest();
        $this->translator = $translator;
    }

    /**
     * @return ItemInterface
     * @throws \Doctrine\ORM\NonUniqueResultException
     */
    public function demande(): ItemInterface
    {
        $routeParams = $this->request->get('_route_params');
        $userName = $this->request->get('user_name');
        $userAcademy = $this->request->get('user_academy');
        $demandeState = $this->request->get('demande_state');
        $demandeEmployee = $this->request->get('demande_whoisincharge');

        $menu = $this->factory->createItem('root');
        $menu->setChildrenAttribute('class', 'menuCatalogue');

        $qb = $this->em->createQueryBuilder();
        if (UserType::TEACHER === $routeParams['user_type']) {
            $qb->select('COUNT(o)')
                ->from('AppBundle:OperationITDLC', 'o')
                ->leftJoin('o.prof', 'u')
                ->where('o.type = :o_type')
                ->setParameter('o_type', OperationITDLCType::DEMANDE);
            if (!empty($userAcademy)) {
                $qb
                    ->andWhere('u.academy = :u_academy')
                    ->setParameter('u_academy', $userAcademy);
            }
            if ($demandeState) {
                $qb
                    ->andWhere('o.state = :o_state')
                    ->setParameter('o_state', $demandeState);
            }
            if ($demandeEmployee) {
                $qb
                    ->andWhere('o.whoIsInCharge = :o_whoisincharge')
                    ->setParameter('o_whoisincharge', $demandeEmployee);
            }
        } else { // Ingénieur
            $routeParams['user_type'] = UserType::ENGINEER;

            $qb->select('COUNT(u)')
                ->from('AppBundle:User', 'u')
                ->where('u.nbContribution != :u_nb_contributions')
                ->andWhere($qb->expr()->isNotNull('u.nbContribution'))
                ->setParameter('u_nb_contributions', 0);
        }

        if (!empty($userName)) {
            $qb->andWhere($qb->expr()->orX(
                $qb->expr()->like(
                    $qb->expr()->upper('u.firstname'),
                    $qb->expr()->upper(':u_firstname')
                ),
                $qb->expr()->like(
                    $qb->expr()->upper('u.lastname'),
                    $qb->expr()->upper(':u_lastname')
                )
            ))
            ->setParameter('u_firstname', "%".$userName."%")
            ->setParameter('u_lastname', "%".$userName."%");
        }

        $nbDemandesItdlc = $qb->getQuery()->getSingleScalarResult();

        $menu
            ->addChild('export', [
                'label' => $this->translator->transchoice(
                    'action.export.'.$routeParams['user_type'],
                    $nbDemandesItdlc,
                    [
                        '%count%' => $nbDemandesItdlc
                    ],
                    'action'
                ),
                'route' => 'admin_demande_itdlc_export',
                'routeParameters' => array_merge(['user_type' => $routeParams['user_type']], $this->request->query->all()),
            ])
            ->setLinkAttribute('class', 'btn btn-export')
        ;

        return $menu;
    }

    /**
     * @return ItemInterface
     */
    public function operation(): ItemInterface
    {
        $menu = $this->factory->createItem('root');
        $menu->setChildrenAttribute('class', 'menuCatalogue');

        $menu
            ->addChild('add', [
                'label' => $this->translator->trans('action.add.operationItdlc.label', [], 'action'),
                'route' => 'admin_operation_itdlc_add',
            ])
            ->setLinkAttribute('class', 'btn btn-add')
        ;

        return $menu;
    }
}
