<?php

namespace AppBundle\Mail;

use AppBundle\Mail\Model\Mail;
use FOS\UserBundle\Mailer\MailerInterface;
use FOS\UserBundle\Model\UserInterface;
use Symfony\Component\Routing\Generator\UrlGeneratorInterface;

/**
 * Custom Mailer for FOSUserBundle.
 * Mails are sent using the SendinBlue provider.
 * 
 * @see https://symfony.com/doc/master/bundles/FOSUserBundle/emails.html#using-a-custom-mailer FOSUserBundle implementation tutorial
 */
class UserMailer extends AbstractMailer implements MailerInterface
{
    /**
     *
     * @param  UserInterface $user
     * @throws \Exception
     */
    public function sendConfirmationEmailMessage(UserInterface $user)
    {
       $url = $this->router->generate(
            'fos_user_registration_confirm',
            ['token' => $user->getConfirmationToken()],
            UrlGeneratorInterface::ABSOLUTE_URL
        );

        $rendered = $this->renderView('User/Mail/registration.html.twig', array(
            'user' => $user,
            'confirmationUrl' =>  $url
        ));

        $this->sendEmailMessage($rendered, $user);
    }

    /**
     *
     * @param  UserInterface $user
     * @throws \Exception
     */
    public function sendResettingEmailMessage(UserInterface $user)
    {
        $url = $this->router->generate(
            'fos_user_resetting_reset',
            ['token' => $user->getConfirmationToken()],
            UrlGeneratorInterface::ABSOLUTE_URL
        );

        $rendered = $this->renderView('User/Mail/resetting.html.twig', array(
            'user' => $user,
            'confirmationUrl' => $url
        ));

        $this->sendEmailMessage($rendered, $user);
    }

    /**
     * @param string $renderedTemplate
     * @param UserInterface $user
     * @throws \Exception
     */
    protected function sendEmailMessage(string $renderedTemplate, UserInterface $user)
    {
        // Render the email, use the first line as the subject, and the rest as the body
        $renderedLines = explode("\n", trim($renderedTemplate));
        $subject = $renderedLines[0];
        $html = implode("\n", array_slice($renderedLines, 1));

        $mailUser = new Mail($subject);
        $mailUser->setFrom($this->getParameter('registration'), $this->getParameter('label'));
        $mailUser->addTo($user->getEmail(), $user->getFullname());
        $mailUser->setHtml($html);
        $mailUser->setText(strip_tags(str_replace('</p><p>', "\n\n", $html))); // Basic HTML cleanup

        $this->send($mailUser);
    }


    /**
     * Send an email to an engineer without solicitations
     * @param UserInterface $user
     * @return array
     * @throws \Exception
     */
    public function sendEmailEngineerWithoutSolicitations($user): array
    {
        $mailEngineer = new Mail(
            sprintf('[%s] suite à votre inscription', $this->getParameter('label'))
        );

        $mailEngineer->setFrom($this->getParameter('registration'), $this->getParameter('label'));

        // Customize engineer's mail
        $mailEngineer->setDescription($this->translator->trans('word.type.user.engineer.label', [], 'word'));
        $mailEngineer->addTo($user->getEmail(), $user->getFullname());
        $mailEngineer->setHtml(
            $this->renderView('User/Mail/solicitation.html.twig', [
                'user' => $user
            ])
        );

        return $this->send($mailEngineer);
    }
}
