<?php

namespace AppBundle\Mail;

use AppBundle\Entity\DemandePEE;
use AppBundle\Entity\User;
use AppBundle\Enum\DemandePEEState;
use AppBundle\Enum\UserType;
use AppBundle\Mail\Model\Mail;

class PeeMailer extends AbstractMailer
{
    /**
     * Send an email to the teacher if the status of his demande PEE change
     *
     * @param DemandePEE $demandePee
     * @param string $demandePeeStatePrevious Previous status of the demande PEE
     *
     * @return array Mailer response
     * @throws \Exception
     */
    public function changeDemandeStatus(DemandePEE $demandePee, string $demandePeeStatePrevious, bool $whithdrawal = false): array
    {
        $mail = new Mail();
        $teacher = $demandePee->getCuser();

        $mail->setFrom($this->getParameter('pee'), $this->getParameter('label'));
        $mail->addTo($teacher->getEmail(), $teacher->getFullname());

        // According to the demande's status, mail have to be customize in a different way
        switch ($demandePee->getState()) {
            case DemandePEEState::REGISTERED:
                if ($teacher->getBossemail()) {
                    $mail->addCc($teacher->getBossemail());
                }
                $mail->setSubject('[Important] Inscription à lʼopération Professeurs en entreprise');
                break;

            case DemandePEEState::PREREGISTERED:
                $mail->setSubject('[CGénial] Préinscription à lʼopération Professeurs en entreprise');
                break;

            default:
                break;
        }

        $mail->setHtml(
            $this->renderView('Admin/Mail/PEE/changeStateDemande.html.twig', [
                'demandepee_state_previous' => $demandePeeStatePrevious,
                'demandepee' => $demandePee,
                'emailUserHashed' => sha1($demandePee->getCuser()->getEmail()),
                'withdrawal' => $whithdrawal,
                'baseURL' =>  $this->getParameter('domain')
            ])
        );

        return $this->send($mail);
    }

    /**
     * Notify CGénial that the user change the sending type of his identity card.
     *
     * @param  User $user
     * @param  string $sendingType The new sending type
     *
     * @return array Mailer response
     * @throws \Exception
     */
    public function changeIdentityCardSendingType(User $user, string $sendingType): array
    {
        $responses = [];

        if (UserType::TEACHER === $user->getType()) {
            $mailTeacher = new Mail('[CGénial] Changement du mode d’envoi de la pièce d’identité pour ' . $user->getFullname());
            $mailTeacher->setFrom($this->getParameter('pee'), $this->getParameter('label'));
            $mailTeacher->addTo($this->getParameter('pee'), $this->getParameter('label'));
            $mailTeacher->setHtml(
                'Lʼenseignant <strong>'.$user->getFullname().'</strong> a modifié ses préférences d’envoi de pièce d’identité en faveur d’une transmission par <strong>voie '. $sendingType .'</strong>.'
            );

            // Send the mail
            $responses = $this->send($mailTeacher);
        }

        return $responses;
    }

    /**
     * Notify the teacher that his PEE applications (DemandePEE entity) have been confirmed.
     *
     * @param User $user
     * @param array $demandes User's applications
     *
     * @return array Mailer response
     * @throws \Exception
     */
    public function confirmDemandes(User $user, array $demandes): array
    {
        $responses = [];

        if (UserType::TEACHER === $user->getType()) {
            $mailTeacher = new Mail('[CGénial] Enregistrement de demande de participation à l’opération Professeurs en entreprise');
            $mailTeacher->setFrom($this->getParameter('pee'), $this->getParameter('label'));
            $mailTeacher->addTo($user->getEmail(), $user->getFullname());
            
            if ($user->getBossemail()) {
                $mailTeacher->addCc($user->getBossemail());
            }

            $mailTeacher->setHtml(
                $this->renderView('Front/Mail/PEE/addDemande.html.twig', [
                    'user' => $user,
                    'demandes' => $demandes,
                ])
            );

            // Send the mail
            $responses = $this->send($mailTeacher);
        }

        return $responses;
    }

    /**
     * Notify Admin - Operation cancelled
     *
     * @param OperationPEE $ope
     * @param User $user
     *
     * @return array Mailer response
     * @throws \Exception
     */
    public function cancelOperation(OperationPEE $ope, User $user): array
    {
        $responses = [];

        $mailAdmin = new Mail('Annulation d\'un participant à une intervention Professeurs en Entreprise');
        // $mailAdmin->setFrom($user->getEmail(), $user->getFullname());
        $mailAdmin->setFrom($this->getParameter('pee'), $this->getParameter('label'));
        // $mailAdmin->addTo($user->getEmail(), $user->getFullname());
        $mailAdmin->addTo($this->getParameter('pee'), $this->getParameter('label'));

        $mailAdmin->setHtml(
            $this->renderView('Front/Mail/PEE/cancelOperation.html.twig', [
                'user' => $user,
                'ope' => $ope,
            ])
        );

        $responses = $this->send($mailAdmin);   

        return $responses;
    }
}
