<?php

namespace AppBundle\Entity;

use Doctrine\ORM\Mapping as ORM;
use Symfony\Component\Validator\Constraints as Assert;
use Symfony\Component\HttpFoundation\File\UploadedFile;

/**
 * @ORM\Table(name="Content")
 * @ORM\Entity(repositoryClass="AppBundle\Repository\ContentRepository")
 * @ORM\HasLifecycleCallbacks()
 */
class Content
{
    /**
     * @var integer $id
     *
     * @ORM\Column(name="id", type="integer")
     * @ORM\Id
     * @ORM\GeneratedValue(strategy="SEQUENCE")
     * @ORM\SequenceGenerator(sequenceName="content_id_seq", allocationSize=1, initialValue=1)
     */
    private $id;

    /**
     * @var string $title
     *
     * @ORM\Column(name="title", type="string", length=255, nullable=true)
     */
    private $title = null;

    /**
     * @var string $picture
     *
     * @ORM\Column(name="picture", type="string", length=255, nullable=true)
     */
    private $picture;

    /**
     * @var string $metatitle
     *
     * @ORM\Column(name="metatitle", type="text", nullable=true)
     */
    private $metatitle;

    /**
     * @var string $metadescription
     *
     * @ORM\Column(name="metadescription", type="text", nullable=true)
     */
    private $metadescription;

    /**
     * @var string $introduction
     *
     * @ORM\Column(name="introduction", type="text", nullable=true)
     */
    private $introduction;
    
    /**
     * @var string $article
     *
     * @ORM\Column(name="article", type="text", nullable=true)
     */
    private $article;

    /**
     * @var integer $position
     *
     * @ORM\Column(name="position", type="integer", nullable=true)
     */
    private $position = 0;

    /**
     * @var boolean $is_module
     *
     * @ORM\Column(name="is_module", type="boolean", nullable=true)
     */
    private $is_module = true;

    /**
     * @var boolean $is_main
     *
     * @ORM\Column(name="is_main", type="boolean", nullable=true)
     */
    private $is_main = false;

    /**
     * @var boolean $is_actu
     *
     * @ORM\Column(name="is_actu", type="boolean", nullable=true)
     */
    private $is_actu = false;

    /**
     * @var boolean $is_active
     *
     * @ORM\Column(name="is_active", type="boolean", nullable=true)
     */
    private $is_active = true;

    /**
     * @var boolean $featured
     *
     * @ORM\Column(name="featured", type="boolean", nullable=true)
     */
    private $featured = false;

    /**
     * @var date $created
     *
     * @ORM\Column(name="created", type="date", nullable=true)
     */
    private $created;

    /**
     * @var datetime $updated
     *
     * @ORM\Column(name="updated", type="datetime", nullable=true)
     */
    private $updated;


    /**
     * @var date $publish
     *
     * @ORM\Column(name="publish", type="date", nullable=true)
     */
    private $publish;

    /**
     * @var date $unpublish
     *
     * @ORM\Column(name="unpublish", type="date", nullable=true)
     */
    private $unpublish;

    /**
     * @var integer $menu
     *
     * @ORM\ManyToOne(targetEntity="Menu", inversedBy="content")
     * @ORM\JoinColumn(name="menu", referencedColumnName="id", nullable=false)
     */
    private $menu;


    public function __tostring()
    {
        return $this->title;
    }

    /**
     * Get id
     *
     * @return integer
     */
    public function getId()
    {
        return $this->id;
    }

    /**
     * Set title
     *
     * @param string $title
     * 
     * @return Content
     */
    public function setTitle($title)
    {
        $this->title = $title;

        return $this;
    }

    /**
     * Get title
     *
     * @return string
     */
    public function getTitle()
    {
        return $this->title;
    }

    /**
     * Set picture
     *
     * @param string $picture
     * 
     * @return Content
     */
    public function setPicture($picture)
    {
        $this->picture = $picture;

        return $this;
    }

    /**
     * Get picture
     *
     * @return string
     */
    public function getPicture()
    {
        return $this->picture;
    }

    /**
     * Set introduction
     *
     * @param string $introduction
     * 
     * @return Content
     */
    public function setIntroduction($introduction)
    {
        $this->introduction = $introduction;

        return $this;
    }

    /**
     * Get introduction
     *
     * @return string
     */
    public function getIntroduction()
    {
        return $this->introduction;
    }

    /**
     * Set article
     *
     * @param string $article
     * 
     * @return Content
     */
    public function setArticle($article)
    {
        $this->article = $article;

        return $this;
    }

    /**
     * Get article
     *
     * @return string
     */
    public function getArticle()
    {
        return $this->article;
    }

    /**
     * Set position
     *
     * @param int $position
     * 
     * @return Content
     */
    public function setPosition($position)
    {
        $this->position = $position;

        return $this;
    }

    /**
     * Get position
     *
     * @return integer
     */
    public function getPosition()
    {
        return $this->position;
    }

    /**
     * Set is_module
     *
     * @param bool $isModule
     * 
     * @return Content
     */
    public function setIsModule($isModule)
    {
        $this->is_module = $isModule;

        return $this;
    }

    /**
     * Get is_module
     *
     * @return boolean
     */
    public function getIsModule()
    {
        return $this->is_module;
    }

    /**
     * Set menu
     *
     * @param \AppBundle\Entity\Menu $menu
     * 
     * @return Content
     */
    public function setMenu(\AppBundle\Entity\Menu $menu = null)
    {
        $this->menu = $menu;

        return $this;
    }

    /**
     * Get menu
     *
     * @return \AppBundle\Entity\Menu
     */
    public function getMenu()
    {
        return $this->menu;
    }

    /**
     * @ORM\PrePersist
     */
    public function setCreatedValue()
    {
        $this->created = new \DateTime();
    }

    /**
     * @ORM\PostPersist
     */
    public function setUpdatedValue()
    {
        $this->updated = new \DateTime();
    }

    /**
     * Set is_main
     *
     * @param bool $isMain
     * 
     * @return Content
     */
    public function setIsMain($isMain)
    {
        $this->is_main = $isMain;

        return $this;
    }

    /**
     * Get is_main
     *
     * @return boolean
     */
    public function getIsMain()
    {
        return $this->is_main;
    }

    /**
     * Set is_actu
     *
     * @param bool $isActu
     * 
     * @return Content
     */
    public function setIsActu($isActu)
    {
        $this->is_actu = $isActu;

        return $this;
    }

    /**
     * Get is_actu
     *
     * @return boolean
     */
    public function getIsActu()
    {
        return $this->is_actu;
    }

    /**
     * Set created
     *
     * @param \DateTime $created
     * 
     * @return Content
     */
    public function setCreated($created)
    {
        $this->created = $created;

        return $this;
    }

    /**
     * Get created
     *
     * @return \DateTime
     */
    public function getCreated()
    {
        return $this->created;
    }

    /**
     * Set updated
     *
     * @param \DateTime $updated
     * 
     * @return Content
     */
    public function setUpdated($updated)
    {
        $this->updated = $updated;

        return $this;
    }

    /**
     * Get updated
     *
     * @return \DateTime
     */
    public function getUpdated()
    {
        return $this->updated;
    }

    /**
     * Set publish
     *
     * @param \DateTime $publish
     * 
     * @return Content
     */
    public function setPublish($publish)
    {
        $this->publish = $publish;

        return $this;
    }

    /**
     * Get publish
     *
     * @return \DateTime
     */
    public function getPublish()
    {
        return $this->publish;
    }

    /**
     * Set unpublish
     *
     * @param \DateTime $unpublish
     * 
     * @return Content
     */
    public function setUnpublish($unpublish)
    {
        $this->unpublish = $unpublish;

        return $this;
    }

    /**
     * Get unpublish
     *
     * @return \DateTime
     */
    public function getUnpublish()
    {
        return $this->unpublish;
    }

    /*** UPLOAD PICTURE ***/

    private $temp;

    /**
     * @Assert\File(
     *  maxSizeMessage = "L'image ne doit pas dépasser 5Mb.",
     *  maxSize = "6000000",
     *  mimeTypes = {"image/jpg", "image/jpeg", "image/gif", "image/png"},
     *   mimeTypesMessage = "Les images doivent être au format JPG, GIF ou PNG.")
     */
    private $file;

    /**
     * Sets file.
     *
     * @param UploadedFile $file
     */
    public function setFile(UploadedFile $file = null)
    {
        $this->file = $file;
        // check if we have an old image path
        if (is_file($this->getAbsolutePicture())) {
            // store the old name to delete after the update
            $this->temp = $this->getAbsolutePicture();
        } else {
            $this->picture = 'initial';
        }
    }

    /**
     * Get file.
     *
     * @return UploadedFile
     */
    public function getFile()
    {
        return $this->file;
    }

    public function getAbsolutePicture()
    {
        return null === $this->picture
                ? null
                : $this->getUploadRootDir().'/'.$this->picture;
    }

    public function getWebPath()
    {
        return null === $this->picture
                ? null
                : $this->getUploadDir().'/'.$this->picture;
    }

    public function getWebPicture()
    {
        return null === $this->picture
                ? null
                : $this->getUploadDir().'/'.$this->picture;
    }

    protected function getUploadRootDir()
    {
        // the absolute directory path where uploaded
        // documents should be saved
        return __DIR__.'/../../../web/'.$this->getUploadDir();
    }

    protected function getUploadDir()
    {
        // get rid of the __DIR__ so it doesn't screw up
        // when displaying uploaded doc/image in the view.
        return 'uploads/content';
    }

    /**
     * @ORM\PrePersist()
     * @ORM\PreUpdate()
     */
    public function preUpload()
    {
        if (null !== $this->getFile()) {
            $filename = sha1(uniqid(mt_rand(), true));
            
            $this->picture = $filename.'-'.$this->getFile()->getClientOriginalName();
        }
    }

    /**
     * @ORM\PostPersist()
     * @ORM\PostUpdate()
     */
    public function upload()
    {
        if (null === $this->getFile()) {
            return;
        }

        // check if we have an old image
        if (isset($this->temp)) {
            // delete the old image
            unlink($this->temp);
            // clear the temp image path
            $this->temp = null;
        }

        // you must throw an exception here if the file cannot be moved
        // so that the entity is not persisted to the database
        // which the UploadedFile move() method does
        $this->getFile()->move(
            $this->getUploadRootDir(),
            $this->picture
        );

        $this->setFile(null);
    }

    /**
     * @ORM\PreRemove()
     */
    public function storeFilenameForRemove()
    {
        $this->temp = $this->getAbsolutePicture();
    }

    /**
     * @ORM\PostRemove()
     */
    public function removeUpload()
    {
        if (isset($this->temp)) {
            unlink($this->temp);
        }
    }

    /**
     * Set featured
     *
     * @param bool $featured
     * 
     * @return Content
     */
    public function setFeatured($featured)
    {
        $this->featured = $featured;

        return $this;
    }

    /**
     * Get featured
     *
     * @return boolean
     */
    public function getFeatured()
    {
        return $this->featured;
    }

    /**
     * Set is_active
     *
     * @param bool $isActive
     * 
     * @return Content
     */
    public function setIsActive($isActive)
    {
        $this->is_active = $isActive;

        return $this;
    }

    /**
     * Get is_active
     *
     * @return boolean
     */
    public function getIsActive()
    {
        return $this->is_active;
    }

    /**
     * Set metatitle
     *
     * @param string $metatitle
     * 
     * @return Content
     */
    public function setMetatitle($metatitle)
    {
        $this->metatitle = $metatitle;

        return $this;
    }

    /**
     * Get metatitle
     *
     * @return string
     */
    public function getMetatitle()
    {
        return $this->metatitle;
    }

    /**
     * Set metadescription
     *
     * @param string $metadescription
     * 
     * @return Content
     */
    public function setMetadescription($metadescription)
    {
        $this->metadescription = $metadescription;

        return $this;
    }

    /**
     * Get metadescription
     *
     * @return string
     */
    public function getMetadescription()
    {
        return $this->metadescription;
    }
}
