<?php

namespace AppBundle\Command;

use AppBundle\Entity\OperationITDLC;
use AppBundle\Enum\OperationITDLCState;
use AppBundle\Translation\ConstTranslator;
use Doctrine\ORM\EntityManagerInterface;
use FOS\OperationITDLCBundle\Model\OperationITDLCManagerInterface;
use Symfony\Bundle\FrameworkBundle\Command\ContainerAwareCommand;
use Symfony\Component\Console\Helper\ProgressBar;
use Symfony\Component\Console\Input\InputArgument;
use Symfony\Component\Console\Input\InputInterface;
use Symfony\Component\Console\Input\InputOption;
use Symfony\Component\Console\Output\OutputInterface;
use Symfony\Component\Stopwatch\Stopwatch;
use Symfony\Component\Validator\Validator\ValidatorInterface;

/**
 * Update 
 * @author Dimitri KURC <technique@agencehpj.fr>
 */
class CloseOldOperationITDLCCommand extends ContainerAwareCommand
{
    /**
     * Configuration
     */
    protected function configure()
    {
        $this
            ->setName('cgenial:operationitdlc:closebefore')
            ->setDescription("Close all operations before date argument")
            ->addArgument('date', InputArgument::REQUIRED, 'Format : Y-m-d. Close all operations before date argument');
    }

    /**
     *
     * @param InputInterface $input
     * @param OutputInterface $output
     * @throws \Doctrine\ORM\ORMException
     * @throws \Doctrine\ORM\OptimisticLockException
     */
    protected function execute(InputInterface $input, OutputInterface $output)
    {
        $em = $this->getContainer()->get("doctrine.orm.entity_manager");

        $operations = $this->findOperationsITDLC($em, $input->getArgument('date'));

        // Create a new progress bar
        $progress = new ProgressBar($output, count($operations));

        // Start and displays the progress bar
        $progress->start();

        foreach ($operations as $operation) {
            $operation->setState(OperationITDLCState::CANCELLED);

            $em->persist($operation);

            $progress->advance();
        }

        $em->flush();

        // Ensure that the progress bar is at 100%
        $progress->finish();
    }

    /**
     * 
     * @param EntityManagerInterface $em
     * @param ValidatorInterface $validator
     * @param int $date
     * @param bool $force
     * @return array List of operations
     */
    private function findOperationsITDLC(
        EntityManagerInterface $em,
        string $date = "2018-06-30"
    ): array {
        $datetime = new \DateTime($date);
        
        $qb = $em->createQueryBuilder();

        // Find operations from the database where the personal address or zipcode is set
        $qb->select('o')
            ->from(OperationITDLC::class, 'o')
            ->where('o.created <= :o_date')
            ->andWhere('o.state NOT IN (:o_state)')
            ->setParameters([
                'o_date' => $datetime,
                'o_state' => [
                    OperationITDLCState::CANCELLED, 
                    OperationITDLCState::CLOSED
                ]
            ]);

        $operations = $qb->getQuery()->getResult();
        
        return $operations;
    }
}
